#include <QString>
#include <QStringList>
#include <QDateTime>
#include <QDebug>

static void fromXMLTVDate(QString &timestr, QDateTime &dt)
{
    // The XMLTV spec requires dates to either be in UTC/GMT or to specify a
    // valid timezone. We are sticking to the spec and require all grabbers
    // to comply.

    if (timestr.isEmpty())
    {
        qWarning() << "Found empty Date/Time in XMLTV data, ignoring";
        return;
    }

    QStringList split = timestr.split(" ");
    QString ts = split[0];
    QDateTime tmpDT;
    tmpDT.setTimeSpec(Qt::LocalTime);

    // UTC/GMT, just strip
    if (ts.endsWith('Z'))
        ts.truncate(ts.length()-1);

    if (ts.length() == 14)
    {
        tmpDT = QDateTime::fromString(ts, "yyyyMMddHHmmss");
    }
    else if (ts.length() == 12)
    {
        tmpDT = QDateTime::fromString(ts, "yyyyMMddHHmm");
    }
    else if (ts.length() == 8)
    {
        tmpDT = QDateTime::fromString(ts, "yyyyMMdd");
    }
    else if (ts.length() == 6)
    {
        tmpDT = QDateTime::fromString(ts, "yyyyMM");
    }
    else if (ts.length() == 4)
    {
        tmpDT = QDateTime::fromString(ts, "yyyy");
    }

    if (!tmpDT.isValid())
    {
        qWarning() <<
            QString("Ignoring unknown timestamp format: %1")
                .arg(ts);
        return;
    }

    if (split.size() > 1)
    {
        QString tmp = split[1].trimmed();

        // These shouldn't be required and they aren't ISO 8601 but the
        // xmltv spec mentions these and just these so handle them just in
        // case
        if (tmp == "GMT" || tmp == "UTC")
            tmp = "+0000";
        else if (tmp == "BST")
            tmp = "+0100";

        // While this seems like a hack, it's better than what was done before
        QString isoDateString = QString("%1 %2").arg(tmpDT.toString(Qt::ISODate))
                                                .arg(tmp);
	qDebug() << QString("Parsing: %1").arg(isoDateString);
        dt = QDateTime::fromString(isoDateString, Qt::ISODate).toUTC();
	qDebug() << QString("Result: %1, TimeSpec: %2").arg(dt.toString()).arg(dt.timeSpec());
    }

    if (!dt.isValid())
    {
        static bool warned_once_on_implicit_utc = false;
        if (!warned_once_on_implicit_utc)
        {
            qWarning() << "No explicit time zone found, "
                "guessing implicit UTC! Please consider enhancing "
                "the guide source to provice explicit UTC or local "
                "time instead.";
            warned_once_on_implicit_utc = true;
        }
        dt = tmpDT;
    }

    dt.setTimeSpec(Qt::UTC);

//    timestr = MythDate::toString(dt, MythDate::kFilename);
}

int main(int argc, char *argv[])
{
    QDateTime dt;
    QString dstr;

    if (argc != 2)
    {
	fprintf(stderr,"Usage: %s <Date String>\n",argv[0]);
	exit(1);
    }

    dstr = QString(argv[1]);
    fromXMLTVDate(dstr,dt);
    qDebug() << QString("Localtime QDateTime: %1").arg(dt.toLocalTime().toString());
}
