#include "dbhelper.h"

#include "mythcontext.h"

QString DBHelper::dbCreateTable(const QString &dbType, const QString &tableName, bool temporary)
{
    QString retval;

    if (dbType == DBTYPE_MYSQL)
    {
        retval = "CREATE ";
        if (temporary)
           retval += "TEMPORARY ";
        retval += "TABLE IF NOT EXISTS " + tableName;
    }
    else if (dbType == DBTYPE_PGSQL)
    {
        retval = dbDropTable(dbType, tableName) + " CREATE ";
        if (temporary)
           retval += "TEMPORARY ";
        retval += "TABLE " + tableName;
    }

    return retval;
}

QString DBHelper::dbDropTable(const QString &dbType, const QString &tableName)
{
   if (dbType == DBTYPE_MYSQL)
      return QString("DROP TABLE IF EXISTS " + tableName);
   else if (dbType == DBTYPE_PGSQL)
      return QString("select drop_table_if_exists ('" + tableName + "', false);");
   return QString("");
}

QString DBHelper::dbInt(const QString &dbType, const QString &colName, const QString &width, bool nullable, bool unsignedCol, const QString &defaultVal, bool auto_num)
{
      QString retval = colName;

   if (dbType == DBTYPE_MYSQL) {
      retval += " int";
      if (width.length() > 0)
         retval += "(" + width + ")";
      if (unsignedCol)
         retval += " unsigned";
      if (!nullable)
         retval += " NOT NULL";
      if (auto_num)
         retval += " auto_increment";
      else if (defaultVal.length() > 0)
         retval += " default " + defaultVal;
   }
   else if (dbType == DBTYPE_PGSQL) {
      if (auto_num)
         retval += " SERIAL";
      else {
         retval += " INT4";
         if (!nullable)
            retval += " NOT NULL";
         if (defaultVal.length() > 0)
            retval += " default " + defaultVal;
         if (unsignedCol)
            retval += " check(" + colName + " >= 0)";
      }
   }
   
   return retval;
}


QString DBHelper::dbTinyInt(const QString &dbType, const QString &colName, const QString &width, bool nullable, bool unsignedCol, const QString &defaultVal, bool auto_num)
{
      QString retval = colName;

   if (dbType == DBTYPE_MYSQL) {
      retval += " tinyint";
      if (width.length() > 0)
         retval += "(" + width + ")";
      if (unsignedCol)
         retval += " unsigned";
      if (!nullable)
         retval += " NOT NULL";
      if (auto_num)
         retval += " auto_increment";
      else if (defaultVal.length() > 0)
         retval += " default " + defaultVal;
   }
   else if (dbType == DBTYPE_PGSQL) {
      if (auto_num)
         retval += " SERIAL";
      else {
         retval += " INT2";
         if (!nullable)
            retval += " NOT NULL";
         if (defaultVal.length() > 0)
            retval += " default " + defaultVal;
         if (unsignedCol)
            retval += " check(" + colName + " >= 0)";
      }
   }
   
   return retval;
}


QString DBHelper::dbBigInt(const QString &dbType, const QString &colName, const QString &width, bool nullable, bool unsignedCol, const QString &defaultVal, bool auto_num)
{
      QString retval = colName;

   if (dbType == DBTYPE_MYSQL) {
      retval += " bigint";
      if (width.length() > 0)
         retval += "(" + width + ")";
      if (unsignedCol)
         retval += " unsigned";
      if (!nullable)
         retval += " NOT NULL";
      if (auto_num)
         retval += " auto_increment";
      else if (defaultVal.length() > 0)
         retval += " default " + defaultVal;
   }
   else if (dbType == DBTYPE_PGSQL) {
      if (auto_num)
         retval += " SERIAL";
      else {
         retval += " INT8";
         if (!nullable)
            retval += " NOT NULL";
         if (defaultVal.length() > 0)
            retval += " default " + defaultVal;
         if (unsignedCol)
            retval += " check(" + colName + " >= 0)";
      }
   }
   
   return retval;
}


QString DBHelper::dbSmallInt(const QString &dbType, const QString &colName, const QString &width, bool nullable, bool unsignedCol, const QString &defaultVal, bool auto_num)
{
      QString retval = colName;

   if (dbType == DBTYPE_MYSQL) {
      retval += " smallint";
      if (width.length() > 0)
         retval += "(" + width + ")";
      if (unsignedCol)
         retval += " unsigned";
      if (!nullable)
         retval += " NOT NULL";
      if (auto_num)
         retval += " auto_increment";
      else if (defaultVal.length() > 0)
         retval += " default " + defaultVal;
   }
   else if (dbType == DBTYPE_PGSQL) {
      if (auto_num)
         retval += " SERIAL";
      else {
         retval += " INT2";
         if (!nullable)
            retval += " NOT NULL";
         if (defaultVal.length() > 0)
            retval += " default " + defaultVal;
         if (unsignedCol)
            retval += " check(" + colName + " >= 0)";
      }
   }
   
   return retval;
}


QString DBHelper::dbMedInt(const QString &dbType, const QString &colName, const QString &width, bool nullable, bool unsignedCol, const QString &defaultVal, bool auto_num)
{
   QString retval = colName;

   if (dbType == DBTYPE_MYSQL) {
      retval += " mediumint";
      if (width.length() > 0)
         retval += "(" + width + ")";
      if (unsignedCol)
         retval += " unsigned";
      if (!nullable)
         retval += " NOT NULL";
      if (auto_num)
         retval += " auto_increment";
      else if (defaultVal.length() > 0)
         retval += " default " + defaultVal;
   }
   else if (dbType == DBTYPE_PGSQL) {
      if (auto_num)
         retval += " SERIAL";
      else {
         retval += " INT4";
         if (!nullable)
            retval += " NOT NULL";
         if (defaultVal.length() > 0)
            retval += " default " + defaultVal;
         if (unsignedCol)
            retval += " check(" + colName + " >= 0)";
      }
   }
   
   return retval;
}

QString DBHelper::dbFloat(const QString &dbType, const QString &colName, const QString &width, bool nullable, bool unsignedCol, const QString &defaultVal)
{
   QString retval = colName + " float";


   if (width.length() > 0)
      retval += "(" + width + ")";

   if (dbType == DBTYPE_MYSQL) {
      if (unsignedCol)
         retval += " unsigned";
      if (!nullable)
         retval += " NOT NULL";
      else if (defaultVal.length() > 0)
         retval += " default " + defaultVal;
   }
   else if (dbType == DBTYPE_PGSQL) {
      if (!nullable)
         retval += " NOT NULL";
      if (defaultVal.length() > 0)
         retval += " default " + defaultVal;
      if (unsignedCol)
         retval += " check(" + colName + " >= 0)";
   }
   
   return retval;
}

QString DBHelper::dbTimeStamp(const QString &dbType, const QString &colName, const QString &width, bool nullable, const QString &defaultVal)
{
    QString retval = colName;

    retval += " TIMESTAMP";

    if (dbType == DBTYPE_MYSQL) {
        if (width.length() > 0)
            retval += "(" + width + ")";
    }

    if (!nullable)
       retval += " NOT NULL";

    if (defaultVal.length() > 0) {
        if (dbType == DBTYPE_MYSQL) {
            retval += " default " + defaultVal;
        }
        else if (dbType == DBTYPE_PGSQL) {
            if (defaultVal == "'00000000000000'" || defaultVal == "'0000-00-00 00:00:00'")
                retval += " default 'epoch'";
            else
                retval += " default " + defaultVal;
        }
    }
    else
    {
        if (dbType == DBTYPE_PGSQL) {
            retval += " default now() ";
        }
    }

    return retval;
}


QString DBHelper::dbDatetime(const QString &dbType, const QString &colName, bool nullable, const QString &defaultVal)
{
    QString retval = colName;

    if (dbType == DBTYPE_MYSQL) {
        retval += " DATETIME";
    }
    else if (dbType == DBTYPE_PGSQL) {
        retval += " TIMESTAMP";
    }

    if (!nullable)
        retval += " NOT NULL";

    if (defaultVal.length() > 0) {
        if (dbType == DBTYPE_MYSQL) {
            retval += " default " + defaultVal;
        }
        else if (dbType == DBTYPE_PGSQL) {
            if (defaultVal == "'00000000000000'" || defaultVal == "'0000-00-00 00:00:00'")
                retval += " default 'epoch'";
            else
                retval += " default " + defaultVal;
        }
    }
    else
    {
        if (dbType == DBTYPE_PGSQL) {
            retval += " default now() ";
        }
    }


   return retval;
}


QString DBHelper::dbDate(const QString &dbType, const QString &colName, bool nullable, const QString &defaultVal)
{
   QString retval = colName + " DATE";

   if (!nullable)
      retval += " NOT NULL";

   if (defaultVal.length() > 0)
      if (dbType == DBTYPE_MYSQL) {
         retval += " default " + defaultVal;
      }
      else if (dbType == DBTYPE_PGSQL) {
         if (defaultVal == "'00000000'" || defaultVal == "'0000-00-00'")
            retval += " default 'epoch'";
         else
            retval += " default " + defaultVal;
      }

   return retval;
}


QString DBHelper::dbYear(const QString &dbType, const QString &colName, const QString &width, bool nullable, const QString &defaultVal)
{
    QString retval = colName;

    if (dbType == DBTYPE_MYSQL)
    {
        retval += " YEAR";

        if (width.length() > 0)
            retval += "(" + width + ")";

        if (!nullable)
            retval += " NOT NULL";

        if (defaultVal.length() > 0)
            retval += " DEFAULT " + defaultVal;
    }
    else if (dbType == DBTYPE_PGSQL)
    {
        retval += " INT";

        if (!nullable)
            retval += " NOT NULL";

        if (defaultVal.length() > 0)
            retval += " DEFAULT " + defaultVal;

        retval += " check(" + colName + " >= 1901 OR " + colName + " = 0000)";
    }

    return retval;
}

QString DBHelper::dbBlob(const QString &dbType, const QString &colName, bool nullable, const QString &defaultVal)
{
    QString retval = colName;

    if (dbType == DBTYPE_MYSQL)
        retval += " BLOB";
    else if (dbType == DBTYPE_PGSQL)
        retval += " BYTEA";

    if (!nullable)
       retval += " NOT NULL";

    if (defaultVal.length() > 0)
        retval += " DEFAULT " + defaultVal;

    return retval;
}

QString DBHelper::dbText(const QString &dbType, const QString &colName, bool nullable)
{
    QString retval = colName + " TEXT";

    if (!nullable)
       retval += " NOT NULL";


    if (dbType == DBTYPE_PGSQL)
        retval += " DEFAULT ''";

    return retval;
}

QString DBHelper::dbAddIndex(const QString &dbType, const QString &tableName, const QString &indexName, const QString &colList, bool unique)
{
    QString retval = "CREATE ";
    QString s;

    int numCols;
    int colStart;
    int colEnd;
    int widthStart;

    if (dbType == DBTYPE_MYSQL)
    {
        if (unique)
            retval += "UNIQUE ";

        retval += "INDEX " + tableName + "_" + indexName + "_idx ON " + tableName + " (" + colList + ")";
    }
    else if (dbType == DBTYPE_PGSQL) 
    {
        if (unique)
            retval += "UNIQUE ";

        retval += "INDEX " + tableName + "_" + indexName + "_idx ON " + tableName + " (";

        // check to see if there are any "partial columns"
        // if they exist, just use the whole column
        // PGSQL won't do partials for an index
        // but it will index on a function, so there may be a
        // way to implement this, since indexing a varchar(255)
        // is slow
        widthStart = colList.find('(');
        if (-1 == widthStart)
            retval += colList + ",";
        else
        {
            numCols = colList.contains(',') + 1;

            colStart = 0;
            if (1 == numCols)
                colEnd = colList.length();
            else 
                colEnd = colList.find(',');

            for (int i = 0 ; i < numCols; ++i)
            {
                if (colEnd < widthStart)
                {
                    retval += colList.section(',', i, i) + ",";
                } 
                else
                {
                    // this part doesn't work for some reason
                    // so just use the whole field.
                    //retval += "substring(" 
                    //       + colList.mid(colStart, colStart-widthStart-1) 
                    //       + " from 1 for " 
                    //       + colList.mid(widthStart+1, widthStart-colList.find(')', widthStart)) + ",";
                    retval += colList.section(',', i, i).section('(', 0, 0) + ",";
		    //retval += colList.mid(colStart, colStart-widthStart-1) + ",";
                    widthStart = colList.find('(', colEnd);
                    if (-1 == widthStart)
                        widthStart = colList.length() + 1;
                }
                colStart = colEnd + 1;
                colEnd = colList.find(',', colStart);
                if (-1 == colEnd)
                    colEnd = colList.length();
            }
        }

        // strip the trailing comma
        retval = retval.left(retval.length()-1);
        retval += ")";
    }
    else
        retval = "";

    return retval;
}

QString DBHelper::dbDropPKey(const QString &dbType, const QString &tableName)
{
    QString  retval = "ALTER TABLE " + tableName + " DROP ";

    if (dbType == DBTYPE_MYSQL)
        retval += "PRIMARY KEY";
    else if (dbType == DBTYPE_PGSQL)
    {
        retval += "CONSTRAINT " + tableName + "_pkey";
    }

    return retval;
}


QString DBHelper::dbChangeColumn(const QString &dbType, const QString &tableName, const QString &oldColName, const QString &newColDef)
{
    QString retval = "ALTER TABLE " + tableName;
    int idx;
    int idx2;

    if (dbType == DBTYPE_MYSQL)
    {
        retval += " CHANGE " + oldColName + " " + newColDef;
    }
    else if (dbType == DBTYPE_PGSQL)
    {
        QString newColName = newColDef.section(' ', 0, 0);

        retval += " ALTER " + oldColName + " TYPE " + newColDef.section(' ', 1, 1) + ";";
        retval += "ALTER TABLE " + tableName + " ALTER " + oldColName;
        if (newColDef.contains("not null", FALSE))
            retval += " SET NOT NULL;";
        else
            retval += " DROP NOT NULL;";

        idx = newColDef.find("default ", 0, FALSE);
        idx2 = newColDef.find(',', idx);
        if (-1 < idx2)
        {
            retval += "ALTER TABLE " + tableName + " ADD " + newColDef.mid(idx2, newColDef.length() - idx2) + ";";
        }
        else
            idx2 = newColDef.length();

        if (-1 < idx)
        {
            idx += 8;
            retval += "ALTER TABLE " + tableName + " ALTER " + oldColName + " SET DEFAULT " + newColDef.mid(idx, idx2-idx);
        }
    }

    return retval;
}

QString DBHelper::dbReplaceInto(const QString &dbType, const QString &tableName, const QString &fields, const QString &values, unsigned long where)
{
    QString retval;
    int numFields = fields.contains(',') + 1;

    if (dbType == DBTYPE_MYSQL)
    {
        retval = "REPLACE INTO " + tableName + " SET ";
        for (int i = 0; i < numFields ; ++i)
        {
            retval += fields.section(',', i, i) + " = " + values.section(",", i, i) + ", ";
        }
        retval = retval.left(retval.length()-2);
    }
    else if (dbType == DBTYPE_PGSQL)
    {
        // the where value is 1 in fields where 
        // the variable is included in a where clause
        // order is from left to right.
        QString strWhere = " WHERE ";
        QString strValues = " VALUES (";

        for (int i = 0; i < numFields ; ++i)
        {
            if (where % 2)
               strWhere += fields.section(',', i, i) + " = " + values.section(",", i, i) + " AND ";
            where = where >> 1;
        }
        strWhere = strWhere.left(strWhere.length()-4);

        // rumor is that MySQL's replace into function 
        // is really a delete and and insert, so let's do that.
        retval = "delete from " + tableName + strWhere + ";";
        retval += "insert into " + tableName + "(";
        for (int i = 0; i < numFields ; ++i)
        {
            retval += fields.section(',', i, i) + ", ";
            strValues += values.section(",", i, i) + ", ";
        }
        strValues = strValues.left(strValues.length()-2);
        strValues += ")";
        retval = retval.left(retval.length()-2);
        retval += ") " + strValues;
    }

    return retval;
}

QString DBHelper::dbTimetoDOW(const QString &dbType, const QString &colName)
{
    QString retval;

    if (dbType == DBTYPE_MYSQL)
        retval = " DAYOFWEEK(" + colName + ") ";
    else if (dbType == DBTYPE_PGSQL)
        retval = " DATE_PART('DOW', " + colName + " ) ";

    return retval;
}

QString DBHelper::dbTimetoDays(const QString &dbType, const QString &colName)
{
    QString retval;

    if (dbType == DBTYPE_MYSQL)
        retval = " TO_DAYS( " + colName + ") ";
    else if (dbType == DBTYPE_PGSQL)
        retval = "DATE_PART('DAY', DATE_TRUNC('DAY', " + colName + " ))::int4 ";
    return retval;
}

QString DBHelper::dbTimetoSeconds(const QString &dbType, const QString &colName)
{
    QString retval;

if (dbType == DBTYPE_MYSQL)
        retval = " TIME_TO_SEC( " + colName + ") ";
    else if (dbType == DBTYPE_PGSQL)
        retval = " DATE_PART('EPOCH', " + colName 
               + " - DATE_TRUNC('DAY', " + colName + " )) ";

    return retval;
}

QString DBHelper::dbFromUnixTime(const QString &dbType, const QString &colName)
{
    QString retval;

    if (dbType == DBTYPE_MYSQL)
        //retval = " FROM_UNIXTIME(" + colName + ") ";
        retval = " interval " + colName + " second ";
    else if (dbType == DBTYPE_PGSQL)
        retval = " " + colName + " * INTERVAL '1 second'";

    return retval;
}

QString DBHelper::dbHoursMinutes(const QString &dbType, const QString &colName)
{
    QString retval;

    if (dbType == DBTYPE_MYSQL)
        retval = " interval time_format(" + colName + ", '%H:%i') hour_minute ";
    else if (dbType == DBTYPE_PGSQL)
        retval = " TO_TIMESTAMP(TO_CHAR(" + colName + ", 'HH24:MI' ), 'HH24:MI') ";

    return retval;
}

QString DBHelper::dbDateSub(const QString &dbType, const QString &colName1, const QString &colName2)
{
    QString retval;

    if (dbType == DBTYPE_MYSQL)
        retval = " DATE_SUB(" + colName1 + ", " + colName2 + ") ";
    else if (dbType == DBTYPE_PGSQL)
        retval = " " + colName1 + " - " + colName2 + " ";

    return retval;
}


QString DBHelper::dbDateAdd(const QString &dbType, const QString &colName1, const QString &colName2)
{
    QString retval;

    if (dbType == DBTYPE_MYSQL)
        retval = " DATE_ADD(" + colName1 + ", " + colName2 + ") ";
    else if (dbType == DBTYPE_PGSQL)
        retval = " " + colName1 + " + " + colName2 + " ";

    return retval;
}

QString DBHelper::dbCurrentTimestamp(const QString &dbType)
{
    QString retval;

    if (dbType == DBTYPE_MYSQL)
        retval = " CURRENT_TIMESTAMP() ";
    else if (dbType == DBTYPE_PGSQL)
        retval = " NOW() ";

    return retval;
}

QString DBHelper::dbCurrentDate(const QString &dbType)
{
    QString retval;

    if (dbType == DBTYPE_MYSQL)
        retval = " CURRENT_DATE() ";
    else if (dbType == DBTYPE_PGSQL)
        retval = " CURRENT_DATE ";

    return retval;
}

QString DBHelper::dbHex(const QString &dbType, const QString &value)
{
    QString retval;

    if (dbType == DBTYPE_MYSQL)
        retval = " 0x" + value;
    else if (dbType == DBTYPE_PGSQL)
        retval = " X'" + value + "'::int ";

    return retval;
}
