#include <mythtv/mythcontext.h>
#include "libvisualplugin.h"

#if defined(SDL_SUPPORT) && defined(LIBVISUAL_SUPPORT) 

#include <iostream>
using namespace std;


LibVisualPlugin::LibVisualPlugin(long int winid, const char* PluginName)
{
    fps = 60;

    m_pVisBin = 0;
    m_pVisVideo = 0;
    m_pSurface = 0;

    // SDL initialisation
    char SDL_windowhack[32];
    sprintf(SDL_windowhack, "%ld", winid);
    setenv("SDL_WINDOWID", SDL_windowhack, 1);

    if (SDL_Init(SDL_INIT_VIDEO | SDL_INIT_NOPARACHUTE) < 0)
    {
        cerr << "Unable to init SDL\n";
        return;
    }
    SDL_ShowCursor(0);


    // LibVisualPlugin initialisation
    if (m_pVisBin = visual_bin_new())
    {
        visual_bin_set_supported_depth(m_pVisBin, VISUAL_VIDEO_DEPTH_ALL);
        if (m_pVisVideo = visual_video_new())
        {
            if (visual_bin_set_video(m_pVisBin, m_pVisVideo) == VISUAL_OK)
            {
                if (visual_bin_connect_by_names(m_pVisBin, const_cast<char*>(PluginName), 0) == VISUAL_OK)
                {
                    if (visual_input_set_callback(visual_bin_get_input(m_pVisBin), AudioCallback, this) == VISUAL_OK)
                    {
                        visual_bin_switch_set_style(m_pVisBin, VISUAL_SWITCH_STYLE_MORPH);
                        visual_bin_switch_set_automatic(m_pVisBin, true);
                        visual_bin_switch_set_steps(m_pVisBin, 100);
                        visual_bin_realize(m_pVisBin);
                    }
                    else
                    {
                        cerr << "Error connecting LibVisualPlugin 'Input' object to our data source object" << endl;
                    }
                }
                else
                {
                    cerr << "Error connecting LibVisualPlugin 'Plugin' object to 'Bin' object" << endl;
                }
            }
            else
            {
                cerr << "Error connecting LibVisualPlugin 'Video' object to 'Bin' object" << endl;
            }
        }
        else
        {
            cerr << "Error allocating LibVisualPlugin 'Video' object" << endl;
        }
    }
    else
    {
        cerr << "Error allocating LibVisualPlugin 'Bin' object" << endl;
    }

}

LibVisualPlugin::~LibVisualPlugin()
{
    if (m_pVisVideo)
    {
         visual_object_unref(VISUAL_OBJECT(m_pVisVideo));
         m_pVisVideo = 0;
    }

    if (m_pVisBin)
    {
         visual_object_unref(VISUAL_OBJECT(m_pVisBin));
         m_pVisBin = 0;
    }

//    visual_quit();   // This appears to cause memory corruption :-(

    SDL_Quit();

    unsetenv("SDL_WINDOWID");
}

void LibVisualPlugin::resize(const QSize &size)
{
    if (visual_bin_get_depth(m_pVisBin) == VISUAL_VIDEO_DEPTH_GL)
    {
        visual_video_set_depth(m_pVisVideo, VISUAL_VIDEO_DEPTH_GL);

        if (const SDL_VideoInfo* VideoInfo = SDL_GetVideoInfo())
        {
            int VideoFlags = SDL_OPENGL | SDL_GL_DOUBLEBUFFER | SDL_HWPALETTE;
            VideoFlags |= VideoInfo->hw_available ? SDL_HWSURFACE : SDL_SWSURFACE;
            VideoFlags |= VideoInfo->blit_hw ? SDL_HWACCEL : 0;

            SDL_GL_SetAttribute (SDL_GL_DOUBLEBUFFER, 1);

            m_pSurface = SDL_SetVideoMode(size.width(), size.height(), 16, VideoFlags);
        }
        else
        {
            cerr << "Error obtaining SDL video information" << endl;
        }
    }
    else
    {
        m_pSurface = SDL_SetVideoMode(size.width(), size.height(), m_pVisVideo->bpp * 8, 0);
    }

    visual_video_set_dimension(m_pVisVideo, size.width(), size.height());
    visual_video_set_buffer(m_pVisVideo, m_pSurface->pixels);
    visual_video_set_pitch(m_pVisVideo, m_pSurface->pitch);
    visual_bin_sync(m_pVisBin, false);
}


bool LibVisualPlugin::process(VisualNode *node)
{
    if (!node || node->length == 0 || !m_pSurface)
        return true;

    int numSamps = 512;
    if (node->length < 512)
        numSamps = node->length;

    int i = 0;
    for (i = 0; i < numSamps; i++)
    {
        m_Audio[0][i] = node->left[i];
        if (node->right)
            m_Audio[1][i] = node->right[i];
        else
            m_Audio[1][i] = m_Audio[0][i];
    }

    for (; i < 512; i++)
    {
        m_Audio[0][i] = 0;
        m_Audio[1][i] = 0;
    }

    return false;
}

bool LibVisualPlugin::draw(QPainter *, const QColor&)
{
    if (visual_bin_depth_changed(m_pVisBin))
    {
        visual_bin_sync(m_pVisBin, true);
    }

    if (visual_bin_get_depth(m_pVisBin) == VISUAL_VIDEO_DEPTH_GL)
    {
        visual_bin_run(m_pVisBin);
        SDL_GL_SwapBuffers();
    }
    else
    {
        if (SDL_MUSTLOCK(m_pSurface) == SDL_TRUE)
        {
            SDL_LockSurface(m_pSurface);
        }

        visual_video_set_buffer(m_pVisVideo, m_pSurface->pixels);
        visual_bin_run(m_pVisBin);

        if (SDL_MUSTLOCK(m_pSurface) == SDL_TRUE)
        {
            SDL_UnlockSurface(m_pSurface);
        }

        if (VisPalette* pVisPalette = visual_bin_get_palette(m_pVisBin))
        {
            SDL_Color Palette[256];

            for(int i = 0; i < 256; i ++)
            {
                Palette[i].r = pVisPalette->colors[i].r;
                Palette[i].g = pVisPalette->colors[i].g;
                Palette[i].b = pVisPalette->colors[i].b;
            }
            SDL_SetColors(m_pSurface, Palette, 0, 256);
        }
        SDL_Flip(m_pSurface);
    }

    return false;
}

// static member function
int LibVisualPlugin::AudioCallback(VisInput*, VisAudio* audio, void* priv)
{
    LibVisualPlugin* that = static_cast<LibVisualPlugin*>(priv);

    VisBuffer buf;
    visual_buffer_init(&buf, that->m_Audio, 1024, 0);
    visual_audio_samplepool_input(audio->samplepool, &buf, VISUAL_AUDIO_SAMPLE_RATE_44100,
            VISUAL_AUDIO_SAMPLE_FORMAT_S16, VISUAL_AUDIO_SAMPLE_CHANNEL_STEREO);

    return 0;
}

class LibVisualPluginFactory : public VisFactory
{
public:
    LibVisualPluginFactory(const char* PluginName) : m_PluginName(PluginName) {}
    const QString& name() const {static QString name(m_PluginName); return m_PluginName;}
    VisualBase* create(MainVisual*, long int winid) const {return new LibVisualPlugin(winid, m_PluginName);}

protected:
    const QString m_PluginName;
};

static class LibVisualPluginFactoryFactory
{
public:
    LibVisualPluginFactoryFactory()
    {
        char* application[] = {"MythMusic"};
        int argc = 1;
        char** argv[] = {application};
        visual_init(&argc, (char***)&argv);

        const char *PluginName = 0;
        while((PluginName = visual_actor_get_next_by_name(PluginName)))
             new LibVisualPluginFactory(PluginName);
   }
} LibVisualPluginFactoryFactory;

#endif // SDL_SUPPORT && LIBVISUAL_SUPPORT
