#ifndef DBHELPERBASE_H_
#define DBHELPERBASE_H_

#include <qstring.h>

/// These are the helper functions for supporting different db backends

class DBHelperBase {
public:

// these were nullable then unsigned

static const int NO_FLAGS = 0;
static const int NOT_NULL = 1;
static const int UNSIGNED = 2;
static const int AUTO_NUM = 4;

static const int NORMAL = 0;
static const int TINY = 1;
static const int SMALL = 2;
static const int MEDIUM = 4;
static const int BIG = 8;

static const int INT4 = 0;
static const int INT2 = 1;
static const int INT8 = 8;

const QString CreateTable(const QString &tableName, bool temporary = false) const;

const QString TableTypeMyISAM() const;

const QString DropTable(const QString &tableName) const;

const QString GroupByWithoutAggregate(const QString &fieldList) const;

const QString Int(const QString &colName, const QString &width = "", int size = NORMAL, int flags = 0, const QString &defaultVal = "") const;

const QString Float(const QString &colName, const QString &width = "", int flags = 0, const QString &defaultVal = "") const;


const QString TimeStamp(const QString &colName, const QString &width = "", int flags = 0, const QString &defaultVal = "") const;

const QString Datetime(const QString &colName, int flags = 0, const QString &defaultVal = "") const;

const QString Date(const QString &colName, int flags = 0, const QString &defaultVal = "") const;

const QString Year(const QString &colName, const QString &width = "", int flags = 0, const QString &defaultVal = "") const;

const QString GetYear(const QString &colName) const;

const QString Blob(const QString &colName, int flags = 0, const QString &defaultVal = "") const;

const QString Text(const QString &colName, int flags = 0) const;


const QString AddIndex(const QString &tableName, const QString &indexName, const QString &colList, bool unique = false) const;

const QString DropPKey(const QString &tableName) const;

const QString AnalyzeTable(const QString &tableName) const;

const QString ChangeColumn(const QString &tableName, const QString &oldColName, const QString &newColDef) const;


const QString TimetoDOW(const QString &colName) const;

const QString TimetoDays(const QString &colName) const;

const QString TimetoSeconds(const QString &colName) const;

const QString FromUnixTime(const QString &colName) const;

const QString HoursMinutes(const QString &colName) const;

const QString YearThroughMinute(const QString &colName) const;


const QString DateSub(const QString &colName1, const QString &colName2) const;

const QString DateAdd(const QString &colName1, const QString &colName2) const;

const QString CurrentTimestamp() const;

const QString CurrentDate() const;


const QString StrConcat(const QString &strList) const;

const QString MatchRegEx(const QString &str, const QString &regex) const;

const QString NotMatchRegEx(const QString &str, const QString &regex) const;

const QString Hex(const QString &value) const;

const QString NextInsertID
    (const QString &tableName, const QString &colName) const;

const QString LockTable(const QString &tableName) const;

const QString UnlockTables() const;

const QString NameEscape() const;


virtual ~DBHelperBase() {}

protected:

DBHelperBase() {}

virtual const QString dbCreateTable(const QString &tableName, bool temporary) const = 0;

virtual const QString dbTableTypeMyISAM() const = 0;

virtual const QString dbDropTable(const QString &tableName) const = 0;

virtual const QString dbGroupByWithoutAggregate(const QString &fieldList) const = 0;

virtual const QString dbInt(const QString &colName, const QString &width, int size, int flags, const QString &defaultVal) const = 0;

virtual const QString dbFloat(const QString &colName, const QString &width, int flags, const QString &defaultVal) const = 0;


virtual const QString dbTimeStamp(const QString &colName, const QString &width, int flags, const QString &defaultVal) const = 0;

virtual const QString dbDatetime(const QString &colName, int flags, const QString &defaultVal) const = 0;

virtual const QString dbDate(const QString &colName, int flags, const QString &defaultVal) const = 0;

virtual const QString dbYear(const QString &colName, const QString &width, int flags, const QString &defaultVal) const = 0;

virtual const QString dbGetYear(const QString &colName) const = 0;

virtual const QString dbBlob(const QString &colName, int flags, const QString &defaultVal) const = 0;

virtual const QString dbText(const QString &colName, int flags) const = 0;



virtual const QString dbAddIndex(const QString &tableName, const QString &indexName, const QString &colList, bool unique) const = 0;

virtual const QString dbDropPKey(const QString &tableName) const = 0;

virtual const QString dbAnalyzeTable(const QString &tableName) const = 0;

virtual const QString dbChangeColumn(const QString &tableName, const QString &oldColName, const QString &newColDef) const = 0;


virtual const QString dbTimetoDOW(const QString &colName) const = 0;

virtual const QString dbTimetoDays(const QString &colName) const = 0;

virtual const QString dbTimetoSeconds(const QString &colName) const = 0;

virtual const QString dbFromUnixTime(const QString &colName) const = 0;

virtual const QString dbHoursMinutes(const QString &colName) const = 0;

virtual const QString dbYearThroughMinute(const QString &colName) const = 0;



virtual const QString dbDateSub(const QString &colName1, const QString &colName2) const = 0;

virtual const QString dbDateAdd(const QString &colName1, const QString &colName2) const = 0;

virtual const QString dbCurrentTimestamp() const = 0;

virtual const QString dbCurrentDate() const = 0;


virtual const QString dbStrConcat(const QString &strList) const = 0;

virtual const QString dbMatchRegEx(const QString &str, const QString &regex) const = 0;

virtual const QString dbNotMatchRegEx(const QString &str, const QString &regex) const = 0;

virtual const QString dbHex(const QString &value) const = 0;

virtual const QString dbNextInsertID
    (const QString &tableName, const QString &colName) const = 0;

virtual const QString dbLockTable(const QString &tableName) const = 0;

virtual const QString dbUnlockTables() const = 0;

virtual const QString dbNameEscape() const = 0;

};

#endif 
