#include "mysqlhelper.h"

#include "mythcontext.h"
#include "mythdbcon.h"

const QString MySQLHelper::dbCreateTable(const QString &tableName, bool temporary) const
{
    QString retval = "CREATE ";

    if (temporary)
       retval += "TEMPORARY ";
    retval += "TABLE IF NOT EXISTS " + tableName;

    return retval;
}

const QString MySQLHelper::dbTableTypeMyISAM() const
{
    return "TYPE=MyISAM";
}

const QString MySQLHelper::dbDropTable(const QString &tableName) const
{
    return QString("DROP TABLE IF EXISTS " + tableName);
}

const QString MySQLHelper::dbGroupByWithoutAggregate(const QString &fieldList) const
{
    return " GROUP BY " + fieldList + " " ;
}

const QString MySQLHelper::dbInt(const QString &colName, const QString &width, int size, int flags, const QString &defaultVal) const
{
   QString retval = colName;

    switch (size)
    {
        case NORMAL:
            retval += " int";
            break;
        case TINY:
            retval += " tinyint";
            break;
        case SMALL:
            retval += " smallint";
            break;
        case MEDIUM:
            retval += " mediumint";
            break;
        case BIG:
            retval += " bigint";
            break;
        default:
            retval += " int";
            break;
    }

    if (width.length() > 0)
        retval += "(" + width + ")";
    if (flags & UNSIGNED)
        retval += " unsigned";
    if (flags & NOT_NULL)
        retval += " NOT NULL";
    if (flags & AUTO_NUM)
        retval += " auto_increment";
    else if (defaultVal.length() > 0)
        retval += " default " + defaultVal;


   return retval;
}


const QString MySQLHelper::dbFloat(const QString &colName, const QString &width, int flags, const QString &defaultVal) const
{
    QString retval = colName + " float";

    if (width.length() > 0)
        retval += "(" + width + ")";

    if (flags & UNSIGNED)
        retval += " unsigned";
    if (flags & NOT_NULL)
        retval += " NOT NULL";
    if (defaultVal.length() > 0)
        retval += " default " + defaultVal;

   return retval;
}

const QString MySQLHelper::dbTimeStamp(const QString &colName, const QString &width, int flags, const QString &defaultVal) const
{
    QString retval = colName + " TIMESTAMP";

    if (width.length() > 0)
        retval += "(" + width + ")";

    if (flags & NOT_NULL)
        retval += " NOT NULL";

    if (defaultVal.length() > 0) {
        retval += " default " + defaultVal;
    }

    return retval;
}


const QString MySQLHelper::dbDatetime(const QString &colName, int flags, const QString &defaultVal) const
{
    QString retval = colName + " DATETIME";

    if (flags & NOT_NULL)
        retval += " NOT NULL";

    if (defaultVal.length() > 0) {
        retval += " default " + defaultVal;
    }

   return retval;
}


const QString MySQLHelper::dbDate(const QString &colName, int flags, const QString &defaultVal) const
{
    QString retval = colName + " DATE";

    if (flags & NOT_NULL)
        retval += " NOT NULL";

    if (defaultVal.length() > 0)
        retval += " default " + defaultVal;

    return retval;
}


const QString MySQLHelper::dbYear(const QString &colName, const QString &width, int flags, const QString &defaultVal) const
{
    QString retval = colName + " YEAR";

    if (width.length() > 0)
        retval += "(" + width + ")";

    if (flags & NOT_NULL)
        retval += " NOT NULL";

    if (defaultVal.length() > 0)
        retval += " DEFAULT " + defaultVal;

    return retval;
}

const QString MySQLHelper::dbGetYear(const QString &colName) const
{
    return colName;
}

const QString MySQLHelper::dbBlob(const QString &colName, int flags, const QString &defaultVal) const
{
    QString retval = colName + " BLOB";

    if (flags & NOT_NULL)
        retval += " NOT NULL";

    if (defaultVal.length() > 0)
        retval += " DEFAULT " + defaultVal;

    return retval;
}

const QString MySQLHelper::dbText(const QString &colName, int flags) const
{
    QString retval = colName + " TEXT";

    if (flags & NOT_NULL)
        retval += " NOT NULL";

    return retval;
}

const QString MySQLHelper::dbAddIndex(const QString &tableName, const QString &indexName, const QString &colList, bool unique) const
{
    QString retval = "CREATE ";
    QString s;

    if (unique)
        retval += "UNIQUE ";

    retval += "INDEX " + tableName + "_" + indexName + "_idx ON " 
           + tableName + " (" + colList + ")";

    return retval;
}

const QString MySQLHelper::dbDropPKey(const QString &tableName) const
{
    return "ALTER TABLE " + tableName + " DROP PRIMARY KEY";
}

const QString MySQLHelper::dbAnalyzeTable(const QString &tableName) const
{
    return "ANALYZE TABLE " + tableName + ";";
}

const QString MySQLHelper::dbChangeColumn(const QString &tableName, const QString &oldColName, const QString &newColDef) const
{
    return "ALTER TABLE " + tableName + " CHANGE " + oldColName 
          + " " + newColDef;
}

const QString MySQLHelper::dbTimetoDOW(const QString &colName) const
{
    return " DAYOFWEEK(" + colName + ") ";
}

const QString MySQLHelper::dbTimetoDays(const QString &colName) const
{
    return " TO_DAYS( " + colName + ") ";
}

const QString MySQLHelper::dbTimetoSeconds(const QString &colName) const
{
    return " TIME_TO_SEC( " + colName + ") ";
}

const QString MySQLHelper::dbFromUnixTime(const QString &colName) const
{
    return " interval " + colName + " second ";
}

const QString MySQLHelper::dbHoursMinutes(const QString &colName) const
{
    return " interval time_format(" + colName + ", '%H:%i') hour_minute ";
}

const QString MySQLHelper::dbYearThroughMinute(const QString &colName) const
{
    return " DATE_FORMAT(" + colName + ", '%Y%m%dT%H%i00') ";
}

const QString MySQLHelper::dbDateSub(const QString &colName1, const QString &colName2) const
{
    return " DATE_SUB(" + colName1 + ", " + colName2 + ") ";
}


const QString MySQLHelper::dbDateAdd(const QString &colName1, const QString &colName2) const
{
    return " DATE_ADD(" + colName1 + ", " + colName2 + ") ";
}

const QString MySQLHelper::dbCurrentTimestamp() const
{
    return " CURRENT_TIMESTAMP() ";
}

const QString MySQLHelper::dbCurrentDate() const
{
    return " CURRENT_DATE() ";
}

const QString MySQLHelper::dbStrConcat(const QString &strList) const
{
    return " CONCAT (" + strList + ") ";
}

const QString MySQLHelper::dbMatchRegEx(const QString &str, const QString &regex) const
{
    return str + " REGEXP " + regex;
}

const QString MySQLHelper::dbNotMatchRegEx(const QString &str, const QString &regex) const
{
    return str + " NOT REGEXP " + regex;
}

const QString MySQLHelper::dbHex(const QString &value) const
{
    return " 0x" + value;
}

const QString MySQLHelper::dbNextInsertID
    (const QString &tableName, const QString &colName) const
{
    MSqlQuery query(MSqlQuery::InitCon());

    QString querystr = QString("INSERT INTO %1 (%2) VALUES (0);")
                               .arg(tableName).arg(colName);

    query.exec(querystr);

    if (!query.isActive() || query.numRowsAffected() < 1) 
    {
        MythContext::DBError("inserting row", query);
        return "-1";
    }

    query.exec("SELECT LAST_INSERT_ID();");

    if (!query.isActive() || query.size() < 1) 
    {
        MythContext::DBError("selecting last insert id", query);
        return "-1";
    }

    query.next();

    return query.value(0).toString();
}

const QString MySQLHelper::dbLockTable(const QString &tableName) const
{
    return "LOCK TABLE " + tableName + " WRITE;";
}

const QString MySQLHelper::dbUnlockTables() const
{
    return "UNLOCK TABLES;";
}

const QString MySQLHelper::dbNameEscape() const
{
    return "`";
}
