#include "pgsqlhelper.h"

#include "mythcontext.h"
#include "mythdbcon.h"

const QString PGSQLHelper::dbCreateTable(const QString &tableName, bool temporary) const
{
    QString retval;

    retval = dbDropTable(tableName) + " CREATE ";
    if (temporary)
       retval += "TEMPORARY ";
    retval += "TABLE " + tableName;

    return retval;
}

const QString PGSQLHelper::dbTableTypeMyISAM() const
{
    return "";
}

const QString PGSQLHelper::dbDropTable(const QString &tableName) const
{
    return QString("select drop_table_if_exists ('" + tableName + "', false);");
}

const QString PGSQLHelper::dbGroupByWithoutAggregate(const QString &fieldList) const
{
    // PGSQL does not support partial GROUP BY clauses.
    return "";
}

const QString PGSQLHelper::dbInt(const QString &colName, const QString &width, int size, int flags, const QString &defaultVal) const
{
    QString retval = colName;

    if (flags & AUTO_NUM)
        retval += " SERIAL";
    else 
    {
        switch (size)
        {
            case NORMAL:
            case MEDIUM:
                retval += " INT4";
                break;
            case TINY:
            case SMALL:
                retval += " INT2";
                break;
            case BIG:
                retval += " INT8";
                break;
            default:
                retval += " INT4";
                break;
        }

        if (flags & NOT_NULL)
            retval += " NOT NULL";
        if (defaultVal.length() > 0)
            retval += " default " + defaultVal;
        if (flags & UNSIGNED)
            retval += " check(" + colName + " >= 0)";
    }

   return retval;
}

const QString PGSQLHelper::dbFloat(const QString &colName, const QString &width, int flags, const QString &defaultVal) const
{
    QString retval = colName + " float";

    if (width.length() > 0)
        retval += "(" + width + ")";

    if (flags & NOT_NULL)
        retval += " NOT NULL";
    if (defaultVal.length() > 0)
        retval += " default " + defaultVal;
    if (flags & UNSIGNED)
        retval += " check(" + colName + " >= 0)";

   return retval;
}

const QString PGSQLHelper::dbTimeStamp(const QString &colName, const QString &width, int flags, const QString &defaultVal) const
{
    QString retval = colName + " TIMESTAMP";

    if (flags & NOT_NULL)
       retval += " NOT NULL";

    if (defaultVal.length() > 0) {
        if (defaultVal == "'00000000000000'" || defaultVal == "'0000-00-00 00:00:00'")
            retval += " default 'epoch'";
        else
            retval += " default " + defaultVal;
    }
    else
    {
        retval += " default now() ";
    }

    return retval;
}


const QString PGSQLHelper::dbDatetime(const QString &colName, int flags, const QString &defaultVal) const
{
    QString retval = colName + " TIMESTAMP(0)";

    if (flags & NOT_NULL)
        retval += " NOT NULL";

    if (defaultVal.length() > 0) {
        if (defaultVal == "'00000000000000'" || defaultVal == "'0000-00-00 00:00:00'")
            retval += " default 'epoch'";
        else
            retval += " default " + defaultVal;
    }
    else
    {
        retval += " default now() ";
    }

   return retval;
}


const QString PGSQLHelper::dbDate(const QString &colName, int flags, const QString &defaultVal) const
{
    QString retval = colName + " DATE";

    if (flags & NOT_NULL)
        retval += " NOT NULL";

    if (defaultVal.length() > 0)
        if (defaultVal == "'00000000'" || defaultVal == "'0000-00-00'")
            retval += " default 'epoch'";
        else
            retval += " default " + defaultVal;

   return retval;
}


const QString PGSQLHelper::dbYear(const QString &colName, const QString &width, int flags, const QString &defaultVal) const
{
    QString retval = colName + " INT";

    if (flags & NOT_NULL)
        retval += " NOT NULL";

    if (defaultVal.length() > 0)
        retval += " DEFAULT " + defaultVal;

    retval += " check(" + colName + " >= 1901 OR " + colName + " = 0000)";

    return retval;
}

const QString PGSQLHelper::dbGetYear(const QString &colName) const
{
    return "cast(" + colName + " AS int)";
}

const QString PGSQLHelper::dbBlob(const QString &colName, int flags, const QString &defaultVal) const
{
    QString retval = colName + " BYTEA";

    if (flags & NOT_NULL)
       retval += " NOT NULL";

    if (defaultVal.length() > 0)
        retval += " DEFAULT " + defaultVal;

    return retval;
}

const QString PGSQLHelper::dbText(const QString &colName, int flags) const
{
    QString retval = colName + " TEXT";

    if (flags & NOT_NULL)
       retval += " NOT NULL";

    retval += " DEFAULT ''";

    return retval;
}

const QString PGSQLHelper::dbAddIndex(const QString &tableName, const QString &indexName, const QString &colList, bool unique) const
{
    QString retval = "CREATE ";
    QString s;

    int numCols;
    int colStart;
    int colEnd;
    int widthStart;

    if (unique)
        retval += "UNIQUE ";

    retval += "INDEX " + tableName + "_" + indexName + "_idx ON " + tableName + " (";

    // check to see if there are any "partial columns"
    // if they exist, just use the whole column
    // PGSQL won't do partials for an index
    // but it will index on a function, so there may be a
    // way to implement this, since indexing a varchar(255)
    // is slow
    widthStart = colList.find('(');
    if (-1 == widthStart)
        retval += colList + ",";
    else
    {
        numCols = colList.contains(',') + 1;

        colStart = 0;
        if (1 == numCols)
            colEnd = colList.length();
        else 
            colEnd = colList.find(',');

        for (int i = 0 ; i < numCols; ++i)
        {
            if (colEnd < widthStart)
            {
                retval += colList.section(',', i, i) + ",";
            } 
            else
            {
                // this part doesn't work for some reason
                // so just use the whole field.
                //retval += "substring(" 
                //       + colList.mid(colStart, colStart-widthStart-1) 
                //       + " from 1 for " 
                //       + colList.mid(widthStart+1, widthStart-colList.find(')', widthStart)) + ",";
                retval += colList.section(',', i, i).section('(', 0, 0) + ",";
                //retval += colList.mid(colStart, colStart-widthStart-1) + ",";
                widthStart = colList.find('(', colEnd);
                if (-1 == widthStart)
                    widthStart = colList.length() + 1;
            }
            colStart = colEnd + 1;
            colEnd = colList.find(',', colStart);
            if (-1 == colEnd)
                colEnd = colList.length();
        }
    }

    // strip the trailing comma
    retval = retval.left(retval.length()-1);
    retval += ")";

    return retval;
}

const QString PGSQLHelper::dbDropPKey(const QString &tableName) const
{
    return "ALTER TABLE " + tableName + " DROP CONSTRAINT " + tableName + "_pkey";
}

const QString PGSQLHelper::dbAnalyzeTable(const QString &tableName) const
{
    return "VACUUM ANALYZE " + tableName + ";";
}

const QString PGSQLHelper::dbChangeColumn(const QString &tableName, const QString &oldColName, const QString &newColDef) const
{
    QString retval = "ALTER TABLE " + tableName;
    int idx;
    int idx2;

    QString newColName = newColDef.section(' ', 0, 0);

    retval += " ALTER " + oldColName + " TYPE " + newColDef.section(' ', 1, 1) + ";";
    retval += "ALTER TABLE " + tableName + " ALTER " + oldColName;
    if (newColDef.contains("not null", FALSE))
        retval += " SET NOT NULL;";
    else
        retval += " DROP NOT NULL;";

    idx = newColDef.find("default ", 0, FALSE);
    idx2 = newColDef.find(',', idx);
    if (-1 < idx2)
    {
        retval += "ALTER TABLE " + tableName + " ADD " + newColDef.mid(idx2, newColDef.length() - idx2) + ";";
    }
    else
        idx2 = newColDef.length();

    if (-1 < idx)
    {
        idx += 8;
        retval += "ALTER TABLE " + tableName + " ALTER " + oldColName + " SET DEFAULT " + newColDef.mid(idx, idx2-idx);
    }

    return retval;
}

const QString PGSQLHelper::dbTimetoDOW(const QString &colName) const
{
    return " DATE_PART('DOW', " + colName + " ) ";
}

const QString PGSQLHelper::dbTimetoDays(const QString &colName) const
{
    return "CAST(DATE_PART('DAY', DATE_TRUNC('DAY', " + colName + " )) AS int4) ";
}

const QString PGSQLHelper::dbTimetoSeconds(const QString &colName) const
{
    return " DATE_PART('EPOCH', " + colName 
               + " - DATE_TRUNC('DAY', " + colName + " )) ";
}

const QString PGSQLHelper::dbFromUnixTime(const QString &colName) const
{
    return " " + colName + " * INTERVAL '1 second'";
}

const QString PGSQLHelper::dbHoursMinutes(const QString &colName) const
{
    return " TO_TIMESTAMP(TO_CHAR(" + colName + ", 'HH24:MI' ), 'HH24:MI') ";
}

const QString PGSQLHelper::dbYearThroughMinute(const QString &colName) const
{
    return " TO_CHAR(" + colName + ", 'YYYYMMDDTHH24MI00') ";
}

const QString PGSQLHelper::dbDateSub(const QString &colName1, const QString &colName2) const
{
    return " " + colName1 + " - " + colName2 + " ";
}


const QString PGSQLHelper::dbDateAdd(const QString &colName1, const QString &colName2) const
{
    return " " + colName1 + " + " + colName2 + " ";
}

const QString PGSQLHelper::dbCurrentTimestamp() const
{
    return " NOW() ";
}

const QString PGSQLHelper::dbCurrentDate() const
{
    return " CURRENT_DATE ";
}

const QString PGSQLHelper::dbStrConcat(const QString &strList) const
{
    QString temp (strList);
    int parens = 0;

    for (unsigned i = 0; i < temp.length(); ++i)
    {
        if (temp[i] == '(')
            ++parens;
        else if (temp[i] == ')')
            --parens;
        else if (temp[i] == ',')
            if (0 == parens)
                   temp.replace(i, 1, " || ");
    }

    return temp;
}

const QString PGSQLHelper::dbMatchRegEx(const QString &str, const QString &regex) const
{
    return str + " ~ " + regex;
}

const QString PGSQLHelper::dbNotMatchRegEx(const QString &str, const QString &regex) const
{
    return str + " !~ " + regex;
}

const QString PGSQLHelper::dbHex(const QString &value) const
{
    return " CAST (X'" + value + "' AS int) ";
}

const QString PGSQLHelper::dbNextInsertID
    (const QString &tableName, const QString &colName) const
{
    MSqlQuery query(MSqlQuery::InitCon());
/*
    query.exec(QString("select seq.relname::text from pg_class src, pg_class seq, "
                       "pg_namespace, pg_attribute,pg_depend where "
                       "pg_depend.refobjsubid = pg_attribute.attnum AND "
                       "pg_depend.refobjid = src.oid AND seq.oid = "
                       "pg_depend.objid AND src.relnamespace = "
                       "pg_namespace.oid AND pg_attribute.attrelid = "
                       "src.oid AND seq.relname like '%1\\\\_%2\\\\_%seq' "
                       "ESCAPE '\\\\';").arg(tableName).arg(colName));
*/

// this assumes only one auto_increment per table...  it is like this to 
// avoid confusion from settings.h classes.

    query.exec(QString("select seq.relname::text from pg_class src, pg_class seq, "
                       "pg_namespace, pg_attribute,pg_depend where "
                       "pg_depend.refobjsubid = pg_attribute.attnum AND "
                       "pg_depend.refobjid = src.oid AND seq.oid = "
                       "pg_depend.objid AND src.relnamespace = "
                       "pg_namespace.oid AND pg_attribute.attrelid = "
                       "src.oid AND seq.relname like '%1\\\\_%seq' "
                       "ESCAPE '\\\\';").arg(tableName));
    if (!query.isActive() || query.size() < 1) 
    {
        MythContext::DBError("selecting last insert id", query);
        return "-1";
    }
    query.next();

    query.exec(QString("select nextval('%1');").arg(query.value(0).toString()));

//    query.exec(QString("select last_value FROM "
//                       "%1;").arg(query.value(0).toString()));

    if (!query.isActive() || query.size() < 1) 
    {
        MythContext::DBError("selecting last insert id", query);
        return "-1";
    }

    query.next();

    return query.value(0).toString();
}


const QString PGSQLHelper::dbLockTable(const QString &tableName) const
{
    return "BEGIN WORK; LOCK TABLE " + tableName + " IN EXCLUSIVE MODE;";
}

const QString PGSQLHelper::dbUnlockTables() const
{
    return "COMMIT WORK;";
}

const QString PGSQLHelper::dbNameEscape() const
{
    return "\"";
}
