#include "pgsqlhelper.h"

#include "mythcontext.h"
#include "mythdbcon.h"

QString PGSQLHelper::dbCreateTable(const QString &tableName, bool temporary)
{
    QString retval;

    retval = dbDropTable(tableName) + " CREATE ";
    if (temporary)
       retval += "TEMPORARY ";
    retval += "TABLE " + tableName;

    return retval;
}

QString PGSQLHelper::dbTableTypeMyISAM()
{
    return "";
}

QString PGSQLHelper::dbDropTable(const QString &tableName)
{
    return QString("select drop_table_if_exists ('" + tableName + "', false);");
}

QString PGSQLHelper::dbGroupByWithoutAggregate(const QString &fieldList)
{
    // PGSQL does not support partial GROUP BY clauses.
    return "";
}

QString PGSQLHelper::dbInt(const QString &colName, const QString &width, bool nullable, bool unsignedCol, const QString &defaultVal, bool auto_num)
{
    QString retval = colName;

    if (auto_num)
        retval += " SERIAL";
    else {
        retval += " INT4";
        if (!nullable)
            retval += " NOT NULL";
        if (defaultVal.length() > 0)
            retval += " default " + defaultVal;
        if (unsignedCol)
            retval += " check(" + colName + " >= 0)";
    }

   return retval;
}


QString PGSQLHelper::dbTinyInt(const QString &colName, const QString &width, bool nullable, bool unsignedCol, const QString &defaultVal, bool auto_num)
{
    QString retval = colName;

    if (auto_num)
        retval += " SERIAL";
    else {
        retval += " INT2";
        if (!nullable)
            retval += " NOT NULL";
        if (defaultVal.length() > 0)
            retval += " default " + defaultVal;
        if (unsignedCol)
            retval += " check(" + colName + " >= 0)";
    }

   return retval;
}


QString PGSQLHelper::dbBigInt(const QString &colName, const QString &width, bool nullable, bool unsignedCol, const QString &defaultVal, bool auto_num)
{
    QString retval = colName;

    if (auto_num)
        retval += " SERIAL";
    else {
        retval += " INT8";
        if (!nullable)
            retval += " NOT NULL";
        if (defaultVal.length() > 0)
            retval += " default " + defaultVal;
        if (unsignedCol)
            retval += " check(" + colName + " >= 0)";
    }

   return retval;
}


QString PGSQLHelper::dbSmallInt(const QString &colName, const QString &width, bool nullable, bool unsignedCol, const QString &defaultVal, bool auto_num)
{
    QString retval = colName;

    if (auto_num)
        retval += " SERIAL";
    else {
        retval += " INT2";
        if (!nullable)
            retval += " NOT NULL";
        if (defaultVal.length() > 0)
            retval += " default " + defaultVal;
        if (unsignedCol)
            retval += " check(" + colName + " >= 0)";
    }

   return retval;
}


QString PGSQLHelper::dbMedInt(const QString &colName, const QString &width, bool nullable, bool unsignedCol, const QString &defaultVal, bool auto_num)
{
    QString retval = colName;

    if (auto_num)
        retval += " SERIAL";
    else {
        retval += " INT4";
        if (!nullable)
            retval += " NOT NULL";
        if (defaultVal.length() > 0)
            retval += " default " + defaultVal;
        if (unsignedCol)
            retval += " check(" + colName + " >= 0)";
    }

   return retval;
}

QString PGSQLHelper::dbFloat(const QString &colName, const QString &width, bool nullable, bool unsignedCol, const QString &defaultVal)
{
    QString retval = colName + " float";

    if (width.length() > 0)
        retval += "(" + width + ")";

    if (!nullable)
        retval += " NOT NULL";
    if (defaultVal.length() > 0)
        retval += " default " + defaultVal;
    if (unsignedCol)
        retval += " check(" + colName + " >= 0)";

   return retval;
}

QString PGSQLHelper::dbTimeStamp(const QString &colName, const QString &width, bool nullable, const QString &defaultVal)
{
    QString retval = colName + " TIMESTAMP";

    if (!nullable)
       retval += " NOT NULL";

    if (defaultVal.length() > 0) {
        if (defaultVal == "'00000000000000'" || defaultVal == "'0000-00-00 00:00:00'")
            retval += " default 'epoch'";
        else
            retval += " default " + defaultVal;
    }
    else
    {
        retval += " default now() ";
    }

    return retval;
}


QString PGSQLHelper::dbDatetime(const QString &colName, bool nullable, const QString &defaultVal)
{
    QString retval = colName + " TIMESTAMP";

    if (!nullable)
        retval += " NOT NULL";

    if (defaultVal.length() > 0) {
        if (defaultVal == "'00000000000000'" || defaultVal == "'0000-00-00 00:00:00'")
            retval += " default 'epoch'";
        else
            retval += " default " + defaultVal;
    }
    else
    {
        retval += " default now() ";
    }

   return retval;
}


QString PGSQLHelper::dbDate(const QString &colName, bool nullable, const QString &defaultVal)
{
    QString retval = colName + " DATE";

    if (!nullable)
        retval += " NOT NULL";

    if (defaultVal.length() > 0)
        if (defaultVal == "'00000000'" || defaultVal == "'0000-00-00'")
            retval += " default 'epoch'";
        else
            retval += " default " + defaultVal;

   return retval;
}


QString PGSQLHelper::dbYear(const QString &colName, const QString &width, bool nullable, const QString &defaultVal)
{
    QString retval = colName + " INT";

    if (!nullable)
        retval += " NOT NULL";

    if (defaultVal.length() > 0)
        retval += " DEFAULT " + defaultVal;

    retval += " check(" + colName + " >= 1901 OR " + colName + " = 0000)";

    return retval;
}

QString PGSQLHelper::dbGetYear(const QString &colName)
{
    return "cast(" + colName + " AS int)";
}

QString PGSQLHelper::dbBlob(const QString &colName, bool nullable, const QString &defaultVal)
{
    QString retval = colName + " BYTEA";

    if (!nullable)
       retval += " NOT NULL";

    if (defaultVal.length() > 0)
        retval += " DEFAULT " + defaultVal;

    return retval;
}

QString PGSQLHelper::dbText(const QString &colName, bool nullable)
{
    QString retval = colName + " TEXT";

    if (!nullable)
       retval += " NOT NULL";

    retval += " DEFAULT ''";

    return retval;
}

QString PGSQLHelper::dbAddIndex(const QString &tableName, const QString &indexName, const QString &colList, bool unique)
{
    QString retval = "CREATE ";
    QString s;

    int numCols;
    int colStart;
    int colEnd;
    int widthStart;

    if (unique)
        retval += "UNIQUE ";

    retval += "INDEX " + tableName + "_" + indexName + "_idx ON " + tableName + " (";

    // check to see if there are any "partial columns"
    // if they exist, just use the whole column
    // PGSQL won't do partials for an index
    // but it will index on a function, so there may be a
    // way to implement this, since indexing a varchar(255)
    // is slow
    widthStart = colList.find('(');
    if (-1 == widthStart)
        retval += colList + ",";
    else
    {
        numCols = colList.contains(',') + 1;

        colStart = 0;
        if (1 == numCols)
            colEnd = colList.length();
        else 
            colEnd = colList.find(',');

        for (int i = 0 ; i < numCols; ++i)
        {
            if (colEnd < widthStart)
            {
                retval += colList.section(',', i, i) + ",";
            } 
            else
            {
                // this part doesn't work for some reason
                // so just use the whole field.
                //retval += "substring(" 
                //       + colList.mid(colStart, colStart-widthStart-1) 
                //       + " from 1 for " 
                //       + colList.mid(widthStart+1, widthStart-colList.find(')', widthStart)) + ",";
                retval += colList.section(',', i, i).section('(', 0, 0) + ",";
                //retval += colList.mid(colStart, colStart-widthStart-1) + ",";
                widthStart = colList.find('(', colEnd);
                if (-1 == widthStart)
                    widthStart = colList.length() + 1;
            }
            colStart = colEnd + 1;
            colEnd = colList.find(',', colStart);
            if (-1 == colEnd)
                colEnd = colList.length();
        }
    }

    // strip the trailing comma
    retval = retval.left(retval.length()-1);
    retval += ")";

    return retval;
}

QString PGSQLHelper::dbDropPKey(const QString &tableName)
{
    return "ALTER TABLE " + tableName + " DROP CONSTRAINT " + tableName + "_pkey";
}

QString PGSQLHelper::dbAnalyzeTable(const QString &tableName)
{
    return "VACUUM ANALYZE " + tableName + ";";
}

QString PGSQLHelper::dbChangeColumn(const QString &tableName, const QString &oldColName, const QString &newColDef)
{
    QString retval = "ALTER TABLE " + tableName;
    int idx;
    int idx2;

    QString newColName = newColDef.section(' ', 0, 0);

    retval += " ALTER " + oldColName + " TYPE " + newColDef.section(' ', 1, 1) + ";";
    retval += "ALTER TABLE " + tableName + " ALTER " + oldColName;
    if (newColDef.contains("not null", FALSE))
        retval += " SET NOT NULL;";
    else
        retval += " DROP NOT NULL;";

    idx = newColDef.find("default ", 0, FALSE);
    idx2 = newColDef.find(',', idx);
    if (-1 < idx2)
    {
        retval += "ALTER TABLE " + tableName + " ADD " + newColDef.mid(idx2, newColDef.length() - idx2) + ";";
    }
    else
        idx2 = newColDef.length();

    if (-1 < idx)
    {
        idx += 8;
        retval += "ALTER TABLE " + tableName + " ALTER " + oldColName + " SET DEFAULT " + newColDef.mid(idx, idx2-idx);
    }

    return retval;
}

QString PGSQLHelper::dbTimetoDOW(const QString &colName)
{
    return " DATE_PART('DOW', " + colName + " ) ";
}

QString PGSQLHelper::dbTimetoDays(const QString &colName)
{
    return "DATE_PART('DAY', DATE_TRUNC('DAY', " + colName + " ))::int4 ";
}

QString PGSQLHelper::dbTimetoSeconds(const QString &colName)
{
    return " DATE_PART('EPOCH', " + colName 
               + " - DATE_TRUNC('DAY', " + colName + " )) ";
}

QString PGSQLHelper::dbFromUnixTime(const QString &colName)
{
    return " " + colName + " * INTERVAL '1 second'";
}

QString PGSQLHelper::dbHoursMinutes(const QString &colName)
{
    return " TO_TIMESTAMP(TO_CHAR(" + colName + ", 'HH24:MI' ), 'HH24:MI') ";
}

QString PGSQLHelper::dbYearThroughMinute(const QString &colName)
{
    return " TO_CHAR(" + colName + ", 'YYYYMMDDTHH24MI00') ";
}

QString PGSQLHelper::dbDateSub(const QString &colName1, const QString &colName2)
{
    return " " + colName1 + " - " + colName2 + " ";
}


QString PGSQLHelper::dbDateAdd(const QString &colName1, const QString &colName2)
{
    return " " + colName1 + " + " + colName2 + " ";
}

QString PGSQLHelper::dbCurrentTimestamp()
{
    return " NOW() ";
}

QString PGSQLHelper::dbCurrentDate()
{
    return " CURRENT_DATE ";
}

QString PGSQLHelper::dbStrConcat(const QString &strList)
{
    QString temp (strList);
    int parens = 0;

    for (int i = 0; i < temp.length(); ++i)
    {
        if (temp[i] == '(')
            ++parens;
        else if (temp[i] == ')')
            --parens;
        else if (temp[i] == ',')
            if (0 == parens)
                   temp.replace(i, 1, " || ");
    }

    return temp;
}

QString PGSQLHelper::dbCharToInt(const QString &colName)
{
    return colName + "::int";
}

QString PGSQLHelper::dbHex(const QString &value)
{
    return " X'" + value + "'::int ";
}

QString PGSQLHelper::dbNextInsertID
    (const QString &tableName, const QString &colName)
{
    MSqlQuery query(MSqlQuery::InitCon());

    query.exec(QString("select seq.relname::text from pg_class src, pg_class seq, "
                       "pg_namespace, pg_attribute,pg_depend where "
                       "pg_depend.refobjsubid = pg_attribute.attnum AND "
                       "pg_depend.refobjid = src.oid AND seq.oid = "
                       "pg_depend.objid AND src.relnamespace = "
                       "pg_namespace.oid AND pg_attribute.attrelid = "
                       "src.oid AND seq.relname like '%1\\\\_%seq' "
                       "ESCAPE '\\\\';").arg(tableName));
    if (!query.isActive() || query.size() < 1) 
    {
        MythContext::DBError("selecting last insert id", query);
        return "-1";
    }
    query.next();

    query.exec(QString("select nextval('%1');").arg(query.value(0).toString()));

//    query.exec(QString("select last_value FROM "
//                       "%1;").arg(query.value(0).toString()));

    if (!query.isActive() || query.size() < 1) 
    {
        MythContext::DBError("selecting last insert id", query);
        return "-1";
    }

    query.next();

    return query.value(0).toString();
}


QString PGSQLHelper::dbLockTable(const QString &tableName)
{
    return "BEGIN WORK; LOCK TABLE " + tableName + " IN EXCLUSIVE MODE;";
}

QString PGSQLHelper::dbUnlockTables()
{
    return "COMMIT WORK;";
}
