/*****************************************************************************
 * = NAME
 * audiooutputca.cpp
 *
 * = DESCRIPTION
 * Core Audio glue for Mac OS X.
 * This plays MythTV audio through the default output device on OS X.
 *
 * = REVISION
 * $Id: audiooutputca.cpp 11165 2006-09-12 10:45:06Z nigel $
 *
 * = AUTHORS
 * Jeremiah Morris
 *****************************************************************************/

#include <CoreServices/CoreServices.h>
#include <CoreAudio/CoreAudio.h>
#include <AudioUnit/AudioUnit.h>

using namespace std;

#include "mythcontext.h"
#include "audiooutputca.h"
#include "config.h"

// this holds Core Audio member variables
class CoreAudio_impl {
    public:
    CoreAudio_impl(AudioOutputCA *inSelf);
    
        // Volume control
        int GetVolumeChannel(int channel); // Returns 0-100
        void SetVolumeChannel(int channel, int volume); // range 0-100 for vol

        // You need to implement the following functions
        bool OpenDevice(void);
        bool OpenAnalogDevice(void);
        bool OpenSPDIFDevice(void);
        void CloseDevice(void);
        void CloseSPDIF(void);

        int getBufferedOnSoundcard(void);

        void Error(QString msg)
        {
            VERBOSE(VB_IMPORTANT, "AudioOutput Error: " + msg);
        }

        void Warn(QString msg)
        {
            VERBOSE(VB_IMPORTANT, "AudioOutput Warning: " + msg);
        }
    
        void ReleaseHogMode();
        bool FindAC3Stream();
        void ResetAudioDevices();
        
    AudioUnit mOutputUnit;

    AudioDeviceID               mSelectedDevice; /* Keeps DeviceID of the selected device */

    /* CoreAudio SPDIF mode specific */
    bool                        mDigitalInUse;      /* Digital (SPDIF) output in use */
    pid_t                       mHogModePID;      /* The keep the pid of our hog status */
    AudioStreamID               mStreamID;    /* The StreamID that has a cac3 streamformat */
    int                         mStreamIndex; /* The index of mStreamID in an AudioBufferList */
    AudioStreamBasicDescription mStreamFormat;  /* The format we changed the stream to */
    AudioStreamBasicDescription mStreamFmtRevert;    /* The original format of the stream */
    bool                        mRevert;       /* Wether we need to revert the stream format */
    bool                        mChangedMixing;/* Wether we need to set the mixing mode back */

    int mBufferedBytes;                          /* Data buffered on the soundcard */

    AudioOutputCA   *mSelf;
    
    // callback for delivering audio to output device
    bool RenderAudio(unsigned char *aubuf, int size,
                     unsigned long long timestamp);

    int AudioStreamChangeFormat( AudioStreamID mStreamID, AudioStreamBasicDescription change_format );
};

// This callback communicates with Core Audio.
static OSStatus RenderCallbackAnalog(void *inRefCon,
                       AudioUnitRenderActionFlags *ioActionFlags,
                       const AudioTimeStamp *inTimeStamp,
                       UInt32 inBusNumber,
                       UInt32 inNumberFrames,
                       AudioBufferList *ioData);

static OSStatus RenderCallbackSPDIF( AudioDeviceID inDevice,
                                   const AudioTimeStamp * inNow, 
                                   const void * inInputData,
                                   const AudioTimeStamp * inInputTime, 
                                   AudioBufferList * outOutputData,
                                   const AudioTimeStamp * inOutputTime, 
                                   void * threadGlobals );

/** \class AudioOutputCA
 *  \brief Implements Core Audio (Mac OS X Hardware Abstraction Layer) output.
 */

AudioOutputCA::AudioOutputCA(
    QString laudio_main_device, QString           laudio_passthru_device,
    int     laudio_bits,        int               laudio_channels,
    int     laudio_samplerate,  AudioOutputSource lsource,
    bool    lset_initial_vol,   bool              laudio_passthru)
    : AudioOutputBase(laudio_main_device, laudio_passthru_device,
                      laudio_bits,        laudio_channels,
                      laudio_samplerate,  lsource,
                      lset_initial_vol,   laudio_passthru),
      mPImpl(new CoreAudio_impl(this))
{
    if (laudio_passthru && !mPImpl->FindAC3Stream())
    {
        VERBOSE(VB_AUDIO, "AudioOutputCA::AudioOutputCA - Configured for AC3 passthru but could not find an AC3 output stream");
        Reconfigure(laudio_bits, laudio_channels,
                laudio_samplerate, false);
    }
    else
    {
        Reconfigure(laudio_bits, laudio_channels,
                laudio_samplerate, laudio_passthru);
    }
}

AudioOutputCA::~AudioOutputCA()
{
    KillAudio();
    
    delete mPImpl;
}

bool AudioOutputCA::OpenDevice()
{
    bool deviceOpened = mPImpl->OpenDevice();
    
    if (deviceOpened && internal_vol && set_initial_vol)
    {
        QString controlLabel = gContext->GetSetting("MixerControl", "PCM");
        controlLabel += "MixerVolume";
        SetCurrentVolume(gContext->GetNumSetting(controlLabel, 80));
    }

    return deviceOpened;
}

void AudioOutputCA::CloseDevice()
{
    mPImpl->CloseDevice();
}


/* Object-oriented part of callback */
bool CoreAudio_impl::RenderAudio(unsigned char *aubuf,
                                int size,
                                unsigned long long timestamp)
{
    if (mSelf->pauseaudio || mSelf->killaudio)
    {
        mSelf->audio_actually_paused = true;
        return false;
    }
    
    /* This callback is called when the sound system requests
       data.  We don't want to block here, because that would
       just cause dropouts anyway, so we always return whatever
       data is available.  If we haven't received enough, either
       because we've finished playing or we have a buffer
       underrun, we play silence to fill the unused space.  */

    int written_size = mSelf->GetAudioData(aubuf, size, false);
    if (written_size && (size > written_size))
    {
        // play silence on buffer underrun
        bzero(aubuf + written_size, size - written_size);
    }
    
    /* update audiotime (bufferedBytes is read by getBufferedOnSoundcard) */
    UInt64 nanos = AudioConvertHostTimeToNanos(
                        timestamp - AudioGetCurrentHostTime());
    mBufferedBytes = (int)((nanos / 1000000000.0) *    // secs
                          (mSelf->effdsp / 100.0) *          // samples/sec
                          mSelf->audio_bytes_per_sample);    // bytes/sample
    mSelf->SetAudiotime();
    
    return (written_size > 0);
}

void AudioOutputCA::WriteAudio(unsigned char *aubuf, int size)
{
    (void)aubuf;
    (void)size;
    return;     // unneeded and unused in CA
}

int AudioOutputCA::getSpaceOnSoundcard(void)
{
    return 0;   // unneeded and unused in CA
}

int AudioOutputCA::getBufferedOnSoundcard(void)
{
    return mPImpl->getBufferedOnSoundcard();
}

void AudioOutputCA::StartOutputThread(void)
{
    return;     // no thread for CA
}

void AudioOutputCA::StopOutputThread(void)
{
    return;     // no thread for CA
}

int AudioOutputCA::GetVolumeChannel(int channel)
{
    return mPImpl->GetVolumeChannel(channel);
}

void AudioOutputCA::SetVolumeChannel(int channel, int volume)
{
    mPImpl->SetVolumeChannel(channel, volume);
}

/* This callback provides converted audio data to the default output device. */
OSStatus RenderCallbackAnalog(void *inRefCon,
                       AudioUnitRenderActionFlags *ioActionFlags,
                       const AudioTimeStamp *inTimeStamp,
                       UInt32 inBusNumber,
                       UInt32 inNumberFrames,
                       AudioBufferList *ioData)
{
    (void)inBusNumber;
    (void)inNumberFrames;
    
    CoreAudio_impl *inst = (CoreAudio_impl *)inRefCon;
    
    if (!inst->RenderAudio((unsigned char *)(ioData->mBuffers[0].mData),
                           ioData->mBuffers[0].mDataByteSize,
                           inTimeStamp->mHostTime))
    {
        // play silence if RenderAudio returns false
        bzero(ioData->mBuffers[0].mData, ioData->mBuffers[0].mDataByteSize);
        *ioActionFlags = kAudioUnitRenderAction_OutputIsSilence;
    }
    return noErr;
}


/*****************************************************************************
 * RenderCallbackSPDIF: callback for SPDIF audio output
 *****************************************************************************/
static OSStatus RenderCallbackSPDIF( AudioDeviceID inDevice,
                                    const AudioTimeStamp * inNow, 
                                    const void * inInputData,
                                    const AudioTimeStamp * inInputTime, 
                                    AudioBufferList * outOutputData,
                                    const AudioTimeStamp * inOutputTime, 
                                    void * inRefCon )
{
    (void)inDevice; // unused
    (void)inNow;        // unused
    (void)inInputData;  // unused
    (void)inInputTime;  // unused

    CoreAudio_impl *inst = (CoreAudio_impl *)inRefCon;
    
    if (!inst->RenderAudio((unsigned char *)(outOutputData->mBuffers[inst->mStreamIndex].mData),
                           outOutputData->mBuffers[inst->mStreamIndex].mDataByteSize,
                           inOutputTime->mHostTime))
    {
        // play silence if RenderAudio returns false
        bzero(outOutputData->mBuffers[inst->mStreamIndex].mData, outOutputData->mBuffers[inst->mStreamIndex].mDataByteSize);
    }
    return noErr;
}


CoreAudio_impl::CoreAudio_impl(AudioOutputCA *inSelf) : mSelf(inSelf)
{
    UInt32 i_param_size;
    AudioDeviceID devid_def = -1;
    OSStatus err;
    
    // Create private data
    mOutputUnit = NULL;
    mSelectedDevice = 0;
    mDigitalInUse = false;
    mRevert = false;
    
    /* Reset all the devices to a default 'non-hog' (ie mixable) format
     * if we don't do this we may be unable to find the Default Output device
     * if we crashed last time leaving it stuck in AC-3 mode */
    ResetAudioDevices();
    
    /* Find the ID of the default Device */
    i_param_size = sizeof( AudioDeviceID );
    err = AudioHardwareGetProperty( kAudioHardwarePropertyDefaultOutputDevice,
                                    &i_param_size, &devid_def );
    if( err != noErr )
    {
        VERBOSE(VB_IMPORTANT, QString("CoreAudio_impl:CoreAudio_impl - could not get default audio device: %1").arg(err, 0, 16));
    }
    mSelectedDevice = devid_def;
    VERBOSE(VB_AUDIO, QString("CoreAudio_impl::CoreAudio_impl - default device ID = %1").arg(mSelectedDevice, 0, 16));
}

bool CoreAudio_impl::OpenDevice()
{
    OSStatus                err = noErr;
    UInt32                  i_param_size = 0;

    VERBOSE(VB_AUDIO, QString("CoreAudio_impl::OpenDevice - Device ID = %1").arg(mSelectedDevice, 0, 16));
    mStreamIndex = -1;
    
    i_param_size = sizeof( mHogModePID );
    err = AudioDeviceGetProperty( mSelectedDevice, 0, FALSE,
                                  kAudioDevicePropertyHogMode,
                                  &i_param_size, &mHogModePID );

    if( err != noErr )
    {
        /* This is not a fatal error. Some drivers simply don't support this property */
        VERBOSE( VB_AUDIO, QString("CoreAudio_impl::OpenDevice - could not check whether device is hogged: %1").arg(err, 0,16));
        mHogModePID = -1;
    }

    if( mHogModePID != -1 && mHogModePID != getpid() )
    {
       VERBOSE(VB_IMPORTANT, QString("Selected audio device is exclusively in use by another program - pid = %1").arg(mHogModePID));
       return false;
    }

    bool deviceOpened = false;
    if (mSelf->audio_passthru)
    {
        deviceOpened = OpenSPDIFDevice();
    }
    
    // If we failed to open the SPDIF device then try openning the analog device instead.
    if (!deviceOpened)
    {
        deviceOpened = OpenAnalogDevice();
    }

    return deviceOpened;
}

bool CoreAudio_impl::OpenAnalogDevice()
{
    // Get default output device
    ComponentDescription desc;
    desc.componentType = kAudioUnitType_Output;
    desc.componentSubType = kAudioUnitSubType_DefaultOutput;
    desc.componentManufacturer = kAudioUnitManufacturer_Apple;
    desc.componentFlags = 0;
    desc.componentFlagsMask = 0;
    
    Component comp = FindNextComponent(NULL, &desc);
    if (comp == NULL)
    {
        Error(QString("FindNextComponent failed"));
        return false;
    }
    
    OSStatus err = OpenAComponent(comp, &mOutputUnit);
    if (err)
    {
        Error(QString("OpenAComponent returned %1").arg((long)err));
        return false;
    }
    
    // Attach callback to default output
    AURenderCallbackStruct input;
    input.inputProc = RenderCallbackAnalog;
    input.inputProcRefCon = this;
    
    err = AudioUnitSetProperty(mOutputUnit,
                               kAudioUnitProperty_SetRenderCallback,
                               kAudioUnitScope_Input,
                               0,
                               &input,
                               sizeof(input));
    if (err)
    {
        Error(QString("AudioUnitSetProperty (callback) returned %1")
                      .arg((long)err));
        return false;
    }
    
    // base class does this after OpenDevice, but we need it now
    mSelf->audio_bytes_per_sample = mSelf->audio_channels * mSelf->audio_bits / 8;
    
    // Set up the audio output unit
    AudioStreamBasicDescription conv_in_desc;
    bzero(&conv_in_desc, sizeof(AudioStreamBasicDescription));
    conv_in_desc.mSampleRate       = mSelf->audio_samplerate;
    conv_in_desc.mFormatID         = kAudioFormatLinearPCM;
    conv_in_desc.mFormatFlags      = kLinearPCMFormatFlagIsSignedInteger;
#ifdef WORDS_BIGENDIAN
    conv_in_desc.mFormatFlags     |= kLinearPCMFormatFlagIsBigEndian;
#endif
    conv_in_desc.mBytesPerPacket   = mSelf->audio_bytes_per_sample;
    conv_in_desc.mFramesPerPacket  = 1;
    conv_in_desc.mBytesPerFrame    = mSelf->audio_bytes_per_sample;
    conv_in_desc.mChannelsPerFrame = mSelf->audio_channels;
    conv_in_desc.mBitsPerChannel   = mSelf->audio_bits;
    
    err = AudioUnitSetProperty(mOutputUnit,
                               kAudioUnitProperty_StreamFormat,
                               kAudioUnitScope_Input,
                               0,
                               &conv_in_desc,
                               sizeof(AudioStreamBasicDescription));
    if (err)
    {
        Error(QString("AudioUnitSetProperty returned %1").arg((long)err));
        return false;
    }
    
    // We're all set up - start the audio output unit
    ComponentResult res = AudioUnitInitialize(mOutputUnit);
    if (res)
    {
        Error(QString("AudioUnitInitialize returned %1").arg((long)res));
        return false;
    }
    
    err = AudioOutputUnitStart(mOutputUnit);
    if (err)
    {
        Error(QString("AudioOutputUnitStart returned %1").arg((long)err));
        return false;
    }
    return true;
}

bool CoreAudio_impl::OpenSPDIFDevice()
{
    OSStatus                err = noErr;
    UInt32                  i_param_size = 0, b_mix = 0;
    Boolean                 b_writeable = false;
    AudioStreamID           *p_streams = NULL;
    int                     i = 0, i_streams = 0;
 
    /* Start doing the SPDIF setup proces */

    /* Hog the device */
    i_param_size = sizeof( mHogModePID );
    mHogModePID = getpid() ;
    
    err = AudioDeviceSetProperty( mSelectedDevice, 0, 0, FALSE,
                                  kAudioDevicePropertyHogMode, i_param_size, &mHogModePID );
    
    if( err != noErr )
    {
        VERBOSE( VB_IMPORTANT, QString("CoreAudio_impl::OpenSPDIF - failed to set hogmode: %1").arg(err, 0, 16));
        return false;
    }

    /* Set mixable to false if we are allowed to */
    err = AudioDeviceGetPropertyInfo( mSelectedDevice, 0, FALSE, kAudioDevicePropertySupportsMixing,
                                    &i_param_size, &b_writeable );

    err = AudioDeviceGetProperty( mSelectedDevice, 0, FALSE, kAudioDevicePropertySupportsMixing,
                                    &i_param_size, &b_mix );
                                    
    if( !err && b_writeable )
    {
        b_mix = 0;
        err = AudioDeviceSetProperty( mSelectedDevice, 0, 0, FALSE,
                            kAudioDevicePropertySupportsMixing, i_param_size, &b_mix );
        mChangedMixing = true;
    }
    
    if( err != noErr )
    {
        VERBOSE(VB_IMPORTANT, QString("CoreAudio_impl::OpenSPDIF - failed to set mixmode: %1").arg(err));
        ReleaseHogMode();
        return false;
    }

    /* Get a list of all the streams on this device */
    err = AudioDeviceGetPropertyInfo( mSelectedDevice, 0, FALSE,
                                      kAudioDevicePropertyStreams,
                                      &i_param_size, NULL );
    if( err != noErr )
    {
        VERBOSE( VB_IMPORTANT, QString("CoreAudio_impl::OpenSPDIF - could not get number of streams: %1").arg(err));
        ReleaseHogMode();
        return false;
    }
    
    i_streams = i_param_size / sizeof( AudioStreamID );
    p_streams = (AudioStreamID *)malloc( i_param_size );
    if( p_streams == NULL )
    {
        VERBOSE(VB_IMPORTANT, "CoreAudio_impl::OpenSPDIF - out of memory" );
        ReleaseHogMode();
        return false;
    }
    
    err = AudioDeviceGetProperty( mSelectedDevice, 0, FALSE,
                                    kAudioDevicePropertyStreams,
                                    &i_param_size, p_streams );
    
    if( err != noErr )
    {
        VERBOSE( VB_IMPORTANT, QString("CoreAudio_impl::OpenSPDIF - could not get number of streams: %1").arg(err));
        if( p_streams ) free( p_streams );
        ReleaseHogMode();
        return false;
    }

    for( i = 0; i < i_streams && mStreamIndex < 0 ; i++ )
    {
        /* Find a stream with a cac3 stream */
        AudioStreamBasicDescription *p_format_list = NULL;
        int                         i_formats = 0, j = 0;
        
        /* Retrieve all the stream formats supported by each output stream */
        err = AudioStreamGetPropertyInfo( p_streams[i], 0,
                                          kAudioStreamPropertyPhysicalFormats,
                                          &i_param_size, NULL );
        if( err != noErr )
        {
            VERBOSE(VB_IMPORTANT, QString("CoreAudio_impl::OpenSPDIF - could not get number of streamformats: %1").arg(err));
            continue;
        }
        
        i_formats = i_param_size / sizeof( AudioStreamBasicDescription );
        p_format_list = (AudioStreamBasicDescription *)malloc( i_param_size );
        if( p_format_list == NULL )
        {
            VERBOSE(VB_IMPORTANT, QString("CoreAudio_impl::OpenSPDIF - could not malloc the memory" ));
            continue;
        }
        
        err = AudioStreamGetProperty( p_streams[i], 0,
                                          kAudioStreamPropertyPhysicalFormats,
                                          &i_param_size, p_format_list );
        if( err != noErr )
        {
            VERBOSE(VB_IMPORTANT, QString("CoreAudio_impl::OpenSPDIF - could not get the list of streamformats: %1").arg(err));
            if( p_format_list) free( p_format_list);
            continue;
        }

        /* Check if one of the supported formats is a digital format */
        for( j = 0; j < i_formats; j++ )
        {
            if( p_format_list[j].mFormatID == 'IAC3' ||
                  p_format_list[j].mFormatID == kAudioFormat60958AC3 )
            {
                VERBOSE(VB_AUDIO, "CoreAudio_impl::OpenSPDIF - found digital format");
                mDigitalInUse = true;
                break;
            }
        }
        
        if (!mDigitalInUse)
        {
            ReleaseHogMode();
            return false;       // No AC3 format streams - try and fallback to analog
        }
        
        if( mDigitalInUse )
        {
            /* if this stream supports a digital (cac3) format, then go set it. */
            int i_requested_rate_format = -1;
            int i_current_rate_format = -1;
            int i_backup_rate_format = -1;

            mStreamID = p_streams[i];
            mStreamIndex = i;

            if( mRevert == false )
            {
                /* Retrieve the original format of this stream first if not done so already */
                i_param_size = sizeof( mStreamFmtRevert );
                err = AudioStreamGetProperty( mStreamID, 0,
                                              kAudioStreamPropertyPhysicalFormat,
                                              &i_param_size, 
                                              &mStreamFmtRevert );
                if( err != noErr )
                {
                    VERBOSE(VB_IMPORTANT, QString("CoreAudio_impl::OpenSPDIF - could not retrieve the original streamformat: %1").arg(err));
                    continue; 
                }
                mRevert = true;
            }

            for( j = 0; j < i_formats; j++ )
            {
                if( p_format_list[j].mFormatID == 'IAC3' ||
                      p_format_list[j].mFormatID == kAudioFormat60958AC3 )
                {
                    if( p_format_list[j].mSampleRate == mSelf->audio_samplerate )
                    {
                        i_requested_rate_format = j;
                        break;
                    }
                    else if( p_format_list[j].mSampleRate == mStreamFmtRevert.mSampleRate )
                    {
                        i_current_rate_format = j;
                    }
                    else
                    {
                        if( i_backup_rate_format < 0 || p_format_list[j].mSampleRate > p_format_list[i_backup_rate_format].mSampleRate )
                            i_backup_rate_format = j;
                    }
                }
                    
            }
            
            if( i_requested_rate_format >= 0 ) /* We prefer to output at the samplerate of the original audio */
                mStreamFormat = p_format_list[i_requested_rate_format];
            else if( i_current_rate_format >= 0 ) /* If not possible, we will try to use the current samplerate of the device */
                mStreamFormat = p_format_list[i_current_rate_format];
            else mStreamFormat = p_format_list[i_backup_rate_format]; /* And if we have to, any digital format will be just fine (highest rate possible) */
        }
        if( p_format_list ) free( p_format_list );
    }
    if( p_streams ) free( p_streams );

    if( !AudioStreamChangeFormat( mStreamID, mStreamFormat ) )
    {
        ReleaseHogMode();
        return false;
    }

    /* Add IOProc callback */
    err = AudioDeviceAddIOProc( mSelectedDevice,
                                (AudioDeviceIOProc)RenderCallbackSPDIF,
                                (void *)this );
    if( err != noErr )
    {
        VERBOSE(VB_IMPORTANT, QString("CoreAudio_impl::OpenSPDIF - AudioDeviceAddIOProc failed: %1").arg(err));
        ReleaseHogMode();
        return false;
    }

    /* Start device */
    err = AudioDeviceStart( mSelectedDevice, (AudioDeviceIOProc)RenderCallbackSPDIF ); 
    if( err != noErr )
    {
        VERBOSE(VB_IMPORTANT, QString("CoreAudio_impl::OpenSPDIF - AudioDeviceStart failed: %1").arg(err));

        err = AudioDeviceRemoveIOProc( mSelectedDevice, 
                                       (AudioDeviceIOProc)RenderCallbackSPDIF );
        if( err != noErr )
        {
            VERBOSE(VB_IMPORTANT, QString("CoreAudio_impl::OpenSPDIF - AudioDeviceRemoveIOProc failed: %1").arg(err));
        }
        ReleaseHogMode();
        return false;
    }

    return true;
}

void CoreAudio_impl::CloseDevice()
{
    if (mOutputUnit)
    {
        AudioOutputUnitStop(mOutputUnit);
        AudioUnitUninitialize(mOutputUnit);
        AudioUnitReset(mOutputUnit,
                       kAudioUnitScope_Input, NULL);
        CloseComponent(mOutputUnit);
        mOutputUnit = NULL;
    }
    
    if (mDigitalInUse)
    {
        // Close SPDIF
        CloseSPDIF();
    }
    if (mHogModePID == getpid())
    {
        ReleaseHogMode();
    }
}
    
void CoreAudio_impl::CloseSPDIF()
{
    OSStatus err;
    UInt32 i_param_size;
    
    /* Stop device */
    err = AudioDeviceStop( mSelectedDevice, 
                           (AudioDeviceIOProc)RenderCallbackSPDIF ); 
    if( err != noErr )
    {
        VERBOSE(VB_AUDIO, QString("CoreAudio_impl::CloseSPDIF - AudioDeviceStop failed: %1").arg(err, 0, 16));
    }

    /* Remove IOProc callback */
    err = AudioDeviceRemoveIOProc( mSelectedDevice,
                                   (AudioDeviceIOProc)RenderCallbackSPDIF );
    if( err != noErr )
    {
        VERBOSE(VB_AUDIO, QString("CoreAudio_impl::CloseSPDIF - AudioDeviceRemoveIOProc failed: %1").arg(err, 0, 16));
    }
    
    if( mRevert )
    {
        AudioStreamChangeFormat( mStreamID, mStreamFmtRevert );
    }

    if( mChangedMixing && mStreamFmtRevert.mFormatID != kAudioFormat60958AC3 )
    {
        int b_mix;
        Boolean b_writeable;
        /* Revert mixable to true if we are allowed to */
        err = AudioDeviceGetPropertyInfo( mSelectedDevice, 0, FALSE, kAudioDevicePropertySupportsMixing,
                                    &i_param_size, &b_writeable );

        err = AudioDeviceGetProperty( mSelectedDevice, 0, FALSE, kAudioDevicePropertySupportsMixing,
                                    &i_param_size, &b_mix );
                                    
        if( !err && b_writeable )
        {
            VERBOSE(VB_AUDIO, QString("CoreAudio_impl::CloseSPDIF - mixable is: %1").arg(b_mix));
            b_mix = 1;
            err = AudioDeviceSetProperty( mSelectedDevice, 0, 0, FALSE,
                                kAudioDevicePropertySupportsMixing, i_param_size, &b_mix );
        }

        if( err != noErr )
        {
            VERBOSE(VB_AUDIO, QString("CoreAudio_impl::CloseSPDIF - failed to set mixmode: %d").arg(err, 0, 16));
        }
    }
}

int CoreAudio_impl::getBufferedOnSoundcard(void)
{
    return mBufferedBytes;
}


int CoreAudio_impl::GetVolumeChannel(int channel)
{
    // FIXME: this only returns global volume
    (void)channel;
    Float32 volume;
    if (!AudioUnitGetParameter(mOutputUnit,
                               kHALOutputParam_Volume,
                              kAudioUnitScope_Global,
                              0,
                              &volume))
    {
        return (int)lroundf(volume * 100.0);
    }
    return 0;    // error case
}

void CoreAudio_impl::SetVolumeChannel(int channel, int volume)
{
    // FIXME: this only sets global volume
    (void)channel;
     AudioUnitSetParameter(mOutputUnit,
                           kHALOutputParam_Volume,
                           kAudioUnitScope_Global,
                           0,
                           (volume * 0.01),
                           0);
}

/*****************************************************************************
 * AudioStreamChangeFormat: Change mStreamID to change_format
 *****************************************************************************/
int CoreAudio_impl::AudioStreamChangeFormat( AudioStreamID mStreamID, AudioStreamBasicDescription change_format )
{
    OSStatus            err = noErr;

    VERBOSE(VB_IMPORTANT, QString("AudioStreamChangeFormat changing format for stream %1").arg(mStreamID));
    /* change the format */
    err = AudioStreamSetProperty( mStreamID, 0, 0,
                                  kAudioStreamPropertyPhysicalFormat,
                                  sizeof( AudioStreamBasicDescription ),
                                  &change_format ); 
    if( err != noErr )
    {
        VERBOSE( VB_IMPORTANT, QString("AudioStreamChangeFormat could not set the stream format: %1").arg(err, 0, 16));
        return false;
    }

    return true;
}

void CoreAudio_impl::ReleaseHogMode()
{
    OSStatus err;
    UInt32 i_param_size;
    
    mHogModePID = -1;
    i_param_size = sizeof( mHogModePID );
    err = AudioDeviceSetProperty( mSelectedDevice, 0, 0, FALSE,
                                     kAudioDevicePropertyHogMode, i_param_size, &mHogModePID );
    if (err != noErr)
    {
        VERBOSE(VB_AUDIO, QString("CoreAudio_impl::ReleaseHogMode - release failed %1").arg(err, 0, 16));
    }
    
}

bool CoreAudio_impl::FindAC3Stream()
{
    OSStatus err;
    UInt32 i_param_size;
    AudioStreamID           *p_streams = NULL;
    int                     i = 0, i_streams = 0;
    bool foundAC3Stream = false;
    
    /* Get a list of all the streams on this device */
    err = AudioDeviceGetPropertyInfo( mSelectedDevice, 0, FALSE,
                                      kAudioDevicePropertyStreams,
                                      &i_param_size, NULL );
    if( err != noErr )
    {
        VERBOSE( VB_IMPORTANT, QString("CoreAudio_impl::FindAC3Stream - could not get number of streams: %1").arg(err));
        ReleaseHogMode();
        return false;
    }

    i_streams = i_param_size / sizeof( AudioStreamID );
    p_streams = (AudioStreamID *)malloc( i_param_size );
    if( p_streams == NULL )
    {
        VERBOSE(VB_IMPORTANT, "CoreAudio_impl::FindAC3Stream - out of memory" );
        ReleaseHogMode();
        return false;
    }

    err = AudioDeviceGetProperty( mSelectedDevice, 0, FALSE,
                                    kAudioDevicePropertyStreams,
                                    &i_param_size, p_streams );

    if( err != noErr )
    {
        VERBOSE( VB_IMPORTANT, QString("CoreAudio_impl::FindAC3Stream - could not get number of streams: %1").arg(err));
        if( p_streams ) free( p_streams );
        ReleaseHogMode();
        return false;
    }

    for( i = 0; i < i_streams && !foundAC3Stream ; i++ )
    {
        /* Find a stream with a cac3 stream */
        AudioStreamBasicDescription *p_format_list = NULL;
        int                         i_formats = 0, j = 0;

        /* Retrieve all the stream formats supported by each output stream */
        err = AudioStreamGetPropertyInfo( p_streams[i], 0,
                                          kAudioStreamPropertyPhysicalFormats,
                                          &i_param_size, NULL );
        if( err != noErr )
        {
            VERBOSE(VB_IMPORTANT, QString("CoreAudio_impl::FindAC3Stream - could not get number of streamformats: %1").arg(err));
            continue;
        }

        i_formats = i_param_size / sizeof( AudioStreamBasicDescription );
        p_format_list = (AudioStreamBasicDescription *)malloc( i_param_size );
        if( p_format_list == NULL )
        {
            VERBOSE(VB_IMPORTANT, QString("CoreAudio_impl::FindAC3Stream - could not malloc the memory" ));
            continue;
        }

        err = AudioStreamGetProperty( p_streams[i], 0,
                                          kAudioStreamPropertyPhysicalFormats,
                                          &i_param_size, p_format_list );
        if( err != noErr )
        {
            VERBOSE(VB_IMPORTANT, QString("CoreAudio_impl::FindAC3Stream - could not get the list of streamformats: %1").arg(err));
            if( p_format_list) free( p_format_list);
            continue;
        }

        /* Check if one of the supported formats is a digital format */
        for( j = 0; j < i_formats; j++ )
        {
            if( p_format_list[j].mFormatID == 'IAC3' ||
                  p_format_list[j].mFormatID == kAudioFormat60958AC3 )
            {
                VERBOSE(VB_AUDIO, "CoreAudio_impl::FindAC3Stream - found digital format");
                foundAC3Stream = true;
                break;
            }
        }

        if (p_format_list)
            free(p_format_list);
    }

    if (p_streams)
        free(p_streams);
        
    return foundAC3Stream;
}

// Reset any devices with an AC3 stream back to a Linear PCM so that they can become a default
// output device
void CoreAudio_impl::ResetAudioDevices()
{
    UInt32 i_param_size;
    OSStatus err;
    AudioHardwareGetPropertyInfo(kAudioHardwarePropertyDevices, &i_param_size, NULL);
    AudioDeviceID *p_device_list = (AudioDeviceID*) malloc(i_param_size);
    UInt32 numDevices = i_param_size / sizeof(AudioDeviceID);
    AudioHardwareGetProperty(kAudioHardwarePropertyDevices, &i_param_size, p_device_list);
    UInt32 i=0;
    for (i=0; i < numDevices; i++)
    {
        err = AudioDeviceGetPropertyInfo(p_device_list[i], 0, false, kAudioDevicePropertyStreams, &i_param_size, NULL);
        if (err != noErr)
        {
            VERBOSE(VB_IMPORTANT, QString("CoreAudio_impl::ResetAudioDevices - could not get number of streams: %1 %2").arg(err).arg(err, 0, 16));
            continue;
        }
        AudioStreamID *p_stream_list = (AudioStreamID*)malloc(i_param_size);
        UInt32 numStreams = i_param_size/sizeof(AudioStreamID);
        AudioDeviceGetProperty(p_device_list[i], 0, false, kAudioDevicePropertyStreams, &i_param_size, p_stream_list);
        UInt32 j=0;
        for (j=0; j < numStreams; j++)
        {
            // Find the streams current physical format
            AudioStreamBasicDescription currentFormat;
            i_param_size = sizeof(currentFormat);
            AudioStreamGetProperty(p_stream_list[j], 0, kAudioStreamPropertyPhysicalFormat, &i_param_size, &currentFormat);

            // If it's currently AC-3/SPDIF then reset it to some mixable format
            if ((currentFormat.mFormatID == 'IAC3') || (currentFormat.mFormatID == kAudioFormat60958AC3))
            {
                err = AudioStreamGetPropertyInfo(p_stream_list[j], 0, kAudioStreamPropertyAvailablePhysicalFormats, &i_param_size, NULL);
                if (err != noErr)
                {
                    VERBOSE(VB_IMPORTANT, QString("CoreAudio_impl::ResetAudioDevices - could not get number of physical formats: %1 %2").arg(err).arg(err, 0, 16));
                    continue;
                }
                AudioStreamBasicDescription *p_format_list = (AudioStreamBasicDescription*)malloc(i_param_size);
                err = AudioStreamGetProperty(p_stream_list[j], 0, kAudioStreamPropertyAvailablePhysicalFormats, &i_param_size, p_format_list);
                if (err != noErr)
                {
                    VERBOSE(VB_IMPORTANT, QString("CoreAudio_impl::ResetAudioDevices - could not get list of physical formats: %1 %2").arg(err).arg(err, 0, 16));
                    continue;
                }
                UInt32 numFormats = i_param_size / sizeof(AudioStreamBasicDescription);
                UInt32 formatIndex = 0;
                bool streamReset = false;
                for (formatIndex=0; (formatIndex < numFormats) && !streamReset; formatIndex++ )
                {
                    if (p_format_list[formatIndex].mFormatID == kAudioFormatLinearPCM)
                    {
                        err = AudioStreamSetProperty(p_stream_list[j], NULL, 0, kAudioStreamPropertyPhysicalFormat, sizeof(AudioStreamBasicDescription), &(p_format_list[formatIndex]));
                        if (err != noErr)
                        {
                            VERBOSE(VB_IMPORTANT, QString("CoreAudio_impl::ResetAudioDevices - could not set physical format: %1 %2").arg(err).arg(err, 0, 16));
                            continue;
                        }
                        else
                        {
                            streamReset = true;
                            sleep(1);   // For the change to take effect
                        }
                    }
                }
                free(p_format_list);
            }
        }
        free(p_stream_list);
    }
    free(p_device_list);
}

