#!/usr/bin/perl -w
#
# animaps.pl
#
# This script grabs the source image(s) and provides an animated version of the
# still images back to Myth.
#
# This script uses the maps.xml file as the list of available sources. Please
# take a look at that file for examples of how to format particular sources.
#
################################################################################
use strict;
use Getopt::Std;
use LWP::Simple;
use Date::Manip;
use MapSearch;
use Data::Dumper;
our ($opt_v, $opt_t, $opt_T, $opt_l, $opt_u, $opt_d); 

###
#
# These defines are required by the MythWeather modules.  What is defined here
# will be displayed in the GUI.
#
###
my $name = 'Animated_Regional_Map';
my $version = 0.21;
my $author = 'Lucien Dunning';
my $email = 'ldunning@gmail.com';

##
# Some defaults.
##
my $updateTimeout = 10*60;
my $retrieveTimeout = 30;
my @types = ('amdesc', 'updatetime', 'animatedimage');
##
# Our xml file is local to the directory that the script is run in.
##
my $dir = "./";

##
# Get and parse the command line arguments
##
getopts('Tvtlu:d:');

if (defined $opt_v) {
    print "$name,$version,$author,$email\n";
    exit 0;
}

if (defined $opt_T) {
    print "$updateTimeout,$retrieveTimeout\n";
    exit 0;
}
if (defined $opt_l) {
    MapSearch::AddDescSearch(shift);
    foreach my $result (@{MapSearch::doSearch()}) {
        print "$result->{animation}::$result->{description}\n" if ($result->{animation});
    }
    exit 0;
}

if (defined $opt_t) {
    foreach (@types) {print; print "\n";}
    exit 0;
}

if (defined $opt_d) {
    $dir = $opt_d;
}

my $loc = shift;

if (!defined $loc || $loc eq "") {
    die "Invalid usage";
}

sub animated_split
{
   my ($path) = shift;
   my ($image) = @_;

   my ($animated, $frame_count);

   use Image::Magick;

   $animated=Image::Magick->new;
   $animated->Read($path . $image);
   $animated->Deconstruct();
   for($frame_count=0; $animated->[$frame_count]; $frame_count++)
   {
      $animated->[$frame_count]->Write($path . $frame_count);
   }
   return ($frame_count);
}

##
# If we got to here, then we have all of the right arguments.  We should
# only get one location via the command line options.
#
# First, we need to search through the XML file for the specified location
# and get all of the necessary attributes of that location.
##
MapSearch::AddAniSearch($loc);
my $results = MapSearch::doSearch();
my $base = $results->[0]->{animation};
my $desc = $results->[0]->{description};
my $size = $results->[0]->{imgsize};
my $file = $desc;
$file =~ s/[^a-zA-Z0-9]//g;
my $path = "$dir/$file-";
my $image_count = 0;
my $i=0;
foreach my $image (sort @{$results->[0]->{images}}) {
    ++$image_count;
}
##
# Because Myth does not support animated GIFs, we have to do a little
# hacking if the source is an animated GIF.
#
# Hack to check if there is only one image listed as the source for the
# animated images.  If it is, then we assume it is an animated GIF (it better
# be, darn it!) and we split it in to its individual frames.
##
if (($image_count == 1) && (@{$results->[0]->{images}}[0] =~ /.gif$/))
{
    # Get the Animated GIF
    getstore("$base/" . @{$results->[0]->{images}}[0], $path . "anim.gif");
    # Split it in to its individual parts
    $i = &animated_split($path,"anim.gif");
}
else
{
   foreach my $image (sort @{$results->[0]->{images}}) {
       getstore("$base/$image", $path . "$i");
       ++$i;
   }
}
# assume all the same size, so just check first
if (!$size) {
    use Image::Size;
    my ($x, $y) = imgsize("${path}0");
    $size = "${x}x$y" if ($x && $y);
} 

print "amdesc::$desc\n";
printf "animatedimage::${path}%%1-$i%s\n", ($size && "-$size" || '');
print "updatetime::Last Updated on " . UnixDate("now", "%b %d, %I:%M %p %Z") . "\n";
