/*
	lcdserver.cpp

	Methods for the core lcdserver object
*/


/*
    Command list
 
    SWITCH_TO_TIME
    SWITCH_TO_MUSIC "Artist" "Album" "Track"
    SWITCH_TO_VOLUME "AppName"
    SWITCH_TO_CHANNEL "ChanNum" "Title" "SubTitle"
    SWITCH_TO_NOTHING
    SWITCH_TO_MENU "Text" Checked Selected Scroll Indent ...
        Text is a string
        Checked can be CHECKED, UNCHECKED, NOTCHECKABLE     
        Selected can be TRUE or FALSE
        Scroll can be TRUE or FALSE
        Indent is an integer
        ... repeat as required
        
    SWITCH_TO_GENERIC RowNo Align "Text" "Screen" Scrollable ...
        RowNo is an integer
        Align can be ALIGN_LEFT, ALIGN_RIGHT, ALIGN_CENTERED
        Scrollable can be TRUE or FALSE 
        ... repeat as required
    
    SET_VOLUME_LEVEL <level>
        level is a float between 0.0 and 1.0
    
    SET_CHANNEL_PROGRESS <progress>
        progress is a float between 0.0 and 1.0
    
    SET_MUSIC_PROGRESS "Time" <progress>
        progress is a float between 0.0 and 1.0

    SET_GENERIC_PROGRESS <progress>
        progress is a float between 0.0 and 1.0
    
    UPDATE_LEDS
    
    STOP_ALL
    
    RESET        
         
*/

#include <qstringlist.h>
#include <qregexp.h>
#include <qdir.h>
#include <qapplication.h>
#include <mythtv/util.h>
#include <mythtv/mythcontext.h>
#include <mythtv/lcddevice.h>

#include "lcdserver.h"

const int LCDSERVER_DEBUG = 0;

LCDServer::LCDServer(int port, QString message, int messageTime)
    :QObject()
{
    m_lcd = new LCDProcClient(this);
    if (!m_lcd->SetupLCD())
    {
        delete m_lcd;
        m_lcd = NULL; 
    } 
    
    //  Create the socket to listen to for connections
    m_serverSocket = new LCDServerSocket(port);
    connect(m_serverSocket, SIGNAL(newConnect(QSocket *)),
            this, SLOT(newConnection(QSocket *)));
    connect(m_serverSocket, SIGNAL(endConnect(QSocket *)),
            this, SLOT(endConnection(QSocket *)));
    
    m_lastSocket = NULL;
  
    //  Announce the port we're listening on
    if (LCDSERVER_DEBUG)
        VERBOSE(VB_ALL, QString("LCDServer: is listening on port %1")
                .arg(port));
    
    if (m_lcd)
        m_lcd->setStartupMessage(message, messageTime);
}

void LCDServer::newConnection(QSocket *socket)
{
    connect(socket, SIGNAL(readyRead()), this, SLOT(readSocket()));
            
    if (LCDSERVER_DEBUG)
        VERBOSE(VB_ALL, "LCDServer: new connection");
    
    if (m_lcd)
        m_lcd->switchToTime();
}

void LCDServer::endConnection(QSocket *socket)
{
    socket->close();
    
    if (LCDSERVER_DEBUG)
        VERBOSE(VB_ALL, "LCDServer: close connection");
    
    if (m_lastSocket == socket)
        m_lastSocket = NULL;
}

void LCDServer::readSocket()
{

    QSocket *socket = (QSocket *)sender();
    m_lastSocket = socket;
    
    while(socket->canReadLine())
    {
        QString incoming_data = socket->readLine();
        incoming_data = incoming_data.replace( QRegExp("\n"), "" );
        incoming_data = incoming_data.replace( QRegExp("\r"), "" );
        incoming_data.simplifyWhiteSpace();
        QStringList tokens = parseCommand(incoming_data);
        parseTokens(tokens, socket);
    }
}

QStringList LCDServer::parseCommand(QString &command)
{
    QStringList tokens;
    QString s = "";
    QChar c;
    bool bInString = false;
    
    for (uint x = 0; x < command.length(); x++)
    {
        c = command[x];
        if (!bInString && c == '"')
            bInString = true;
        else if (bInString && c == '"')
            bInString = false;
        else if (!bInString && c == ' ')
        {
            tokens.append(s);
            s = "";
        }    
        else
            s = s + c;
    }
    
    tokens.append(s);
    
    return tokens;
}

void LCDServer::parseTokens(const QStringList &tokens, QSocket *socket)
{
    //
    //  parse commands coming in from the socket
    //
    
    if(tokens[0] == "HALT" ||
       tokens[0] == "QUIT" ||
       tokens[0] == "SHUTDOWN")
    {
        shutDown();
    }
    else if (tokens[0] == "HELLO")
    {
        sendConnected(socket);
    }
    else if (tokens[0] == "SWITCH_TO_TIME")
    {
        switchToTime(socket);
    }
    else if (tokens[0] == "SWITCH_TO_MUSIC")
    {
        switchToMusic(tokens, socket);
    }
    else if (tokens[0] == "SWITCH_TO_VOLUME")
    {
        switchToVolume(tokens, socket);
    }
    else if (tokens[0] == "SWITCH_TO_GENERIC")
    {
        switchToGeneric(tokens, socket);
    }
    else if (tokens[0] == "SWITCH_TO_MENU")
    {
        switchToMenu(tokens, socket);
    }
    else if (tokens[0] == "SWITCH_TO_CHANNEL")
    {
        switchToChannel(tokens, socket);
    }
    else if (tokens[0] == "SWITCH_TO_NOTHING")
    {
        switchToNothing(socket);
    }
    else if (tokens[0] == "SET_VOLUME_LEVEL")
    {
        setVolumeLevel(tokens, socket);
    }
    else if (tokens[0] == "SET_MUSIC_PROGRESS")
    {
        setMusicProgress(tokens, socket);
    }
    else if (tokens[0] == "SET_CHANNEL_PROGRESS")
    {
        setChannelProgress(tokens, socket);
    }
    else if (tokens[0] == "UPDATE_LEDS")
    {
        updateLEDs(tokens, socket);
    }
    else if (tokens[0] == "STOP_ALL")
    {
        if (m_lcd)
            m_lcd->stopAll();
    }
    else if (tokens[0] == "RESET")
    {
        // reload the settings from the database
        if (m_lcd)
            m_lcd->loadSettings();
    }

    else
    {
        QString did_not_parse = tokens.join(" ");
        
        if (LCDSERVER_DEBUG)
            VERBOSE(VB_ALL, "LCDServer::failed to parse this: " << did_not_parse);

        sendMessage(socket, "HUH?");
    }
}

void LCDServer::shutDown()
{
    if (LCDSERVER_DEBUG)
        VERBOSE(VB_ALL, "LCDServer:: shuting down");
    
    delete m_serverSocket;
    
    exit(0);
}

void LCDServer::sendMessage(QSocket *where, const QString &what)
{
    QString message = what;
    message.append("\n");
    where->writeBlock(message.utf8(), message.utf8().length());
}

void LCDServer::sendKeyPress(QString key_pressed)
{
    if (LCDSERVER_DEBUG)
        VERBOSE(VB_ALL, "LCDServer:: send key press: " << key_pressed);

    // send key press to last client that sent us a message
    if (m_lastSocket)
        sendMessage(m_lastSocket, "KEY " + key_pressed);  
}

void LCDServer::sendConnected(QSocket *socket)
{
    QString sWidth, sHeight;
    int nWidth = 0, nHeight = 0;
    
    if (m_lcd)
    {
        nWidth = m_lcd->getLCDWidth();
        nHeight = m_lcd->getLCDHeight();
    }
    
    sWidth = sWidth.setNum(nWidth);
    sHeight = sHeight.setNum(nHeight);
    
    sendMessage(socket, "CONNECTED " + sWidth + " " + sHeight);
}

void LCDServer::switchToTime(QSocket *socket)
{
    if (LCDSERVER_DEBUG)
        VERBOSE(VB_ALL, "LCDServer:: SWITCH_TO_TIME");

    if (m_lcd)
        m_lcd->switchToTime();      

    sendMessage(socket, "OK");
}

void LCDServer::switchToMusic(const QStringList &tokens, QSocket *socket)
{
    if (LCDSERVER_DEBUG)
        VERBOSE(VB_ALL, "LCDServer:: SWITCH_TO_MUSIC");
    
    QString flat = tokens.join(" ");
   
    if (tokens.count() != 4)
    {
        VERBOSE(VB_ALL, "LCDServer: bad SWITCH_TO_MUSIC command: " << flat);
        sendMessage(socket, "HUH?");
        return;
    }
     
    if (m_lcd)
        m_lcd->switchToMusic(tokens[1], tokens[2], tokens[3]);
        
    sendMessage(socket, "OK");
}

void LCDServer::switchToGeneric(const QStringList &tokens, QSocket *socket)
{
    if (LCDSERVER_DEBUG)
        VERBOSE(VB_ALL, "LCDServer: SWITCH_TO_GENERIC");
    
    QString flat = tokens.join(" ");
   
    if ((tokens.count() - 1) % 5 != 0)
    {
        VERBOSE(VB_ALL, "LCDServer: bad no. of args SWITCH_TO_GENERIC "
                        "command: " <<  flat);
        sendMessage(socket, "HUH?");
        return;
    }
     
    QPtrList<LCDTextItem> items;
    items.setAutoDelete(true);
    
    for (uint x = 1; x < tokens.count(); x += 5)
    {
        bool bOK;
        int row = tokens[x].toInt(&bOK);
        if (!bOK)
        {
            VERBOSE(VB_ALL, "LCDServer: bad row no. in SWITCH_TO_GENERIC " 
                    "command: " << tokens[x]);
            sendMessage(socket, "HUH?");
            return;
        }

        TEXT_ALIGNMENT align;
        if (tokens[x + 1] == "ALIGN_LEFT")
            align = ALIGN_LEFT;    
        else if (tokens[x + 1] == "ALIGN_RIGHT") 
            align = ALIGN_RIGHT;
        else if (tokens[x + 1] == "ALIGN_CENTERED")
            align = ALIGN_CENTERED;
        else
        {
            VERBOSE(VB_ALL, "LCDServer: bad align in SWITCH_TO_GENERIC " 
                    "command: " << tokens[x + 1]);
            sendMessage(socket, "HUH?");
            return;
        }
        
        QString text = tokens[x + 2];
        QString screen = tokens[x + 3];
        bool scrollable;
        if (tokens[x + 4] == "TRUE")
            scrollable = true;
        else if (tokens[x + 4] == "FALSE")
            scrollable = false; 
        else
        {
            VERBOSE(VB_ALL, "LCDServer: bad scrollable bool in SWITCH_TO_GENERIC " 
                    "command: " << tokens[x + 4]);
            sendMessage(socket, "HUH?");
            return;
        }
        
        items.append(new LCDTextItem(row, align, text, screen, scrollable));
    }
    
    if (m_lcd)
        m_lcd->switchToGeneric(&items);
        
    sendMessage(socket, "OK");
}

void LCDServer::switchToChannel(const QStringList &tokens, QSocket *socket)
{
    if (LCDSERVER_DEBUG)
        VERBOSE(VB_ALL, "LCDServer: SWITCH_TO_CHANNEL");
    
    QString flat = tokens.join(" ");

    if (tokens.count() != 4)
    {
        VERBOSE(VB_ALL, "LCDServer: bad SWITCH_TO_CHANNEL command: " << flat);
        sendMessage(socket, "HUH?");
        return;
    }
     
    if (m_lcd)
        m_lcd->switchToChannel(tokens[1], tokens[2], tokens[3]);
        
    sendMessage(socket, "OK");
}

void LCDServer::switchToVolume(const QStringList &tokens, QSocket *socket)
{
    if (LCDSERVER_DEBUG)
        VERBOSE(VB_ALL, "LCDServer: SWITCH_TO_VOLUME");
    
    QString flat = tokens.join(" ");
   
    if (tokens.count() != 2)
    {
        VERBOSE(VB_ALL, "LCDServer: bad SWITCH_TO_VOLUME command: " << flat);
        sendMessage(socket, "HUH?");
        return;
    }
     
    if (m_lcd)
        m_lcd->switchToVolume(tokens[1]);
        
    sendMessage(socket, "OK");
}

void LCDServer::switchToNothing(QSocket *socket)
{
    if (LCDSERVER_DEBUG)
        VERBOSE(VB_ALL, "LCDServer: SWITCH_TO_NOTHING");
    
    if (m_lcd)
        m_lcd->switchToNothing();
    
    sendMessage(socket, "OK");        
}

void LCDServer::switchToMenu(const QStringList &tokens, QSocket *socket)
{
    if (LCDSERVER_DEBUG)
        VERBOSE(VB_ALL, "LCDServer: SWITCH_TO_MENU");
    
    QString flat = tokens.join(" ");
   
    if ((tokens.count() - 3) % 5 != 0)
    {
        VERBOSE(VB_ALL, "LCDServer: bad no. of args SWITCH_TO_MENU command: " 
                << flat);
        sendMessage(socket, "HUH?");
        return;
    }
     
    QString appName = tokens[1];
    
    bool bPopup;
    if (tokens[2] == "TRUE")
        bPopup = true;
    else if (tokens[2] == "FALSE")
        bPopup = false; 
    else
    {
        VERBOSE(VB_ALL, "LCDServer: bad popup bool in SWITCH_TO_MENU " 
                        "command: " << tokens[2]);
        sendMessage(socket, "HUH?");
        return;
    }
    
    QPtrList<LCDMenuItem> items;
    items.setAutoDelete(true);
    
    for (uint x = 3; x < tokens.count(); x += 5)
    {
        QString text = tokens[x];

        CHECKED_STATE checked;
        if (tokens[x + 1] == "CHECKED")
            checked = CHECKED;    
        else if (tokens[x + 1] == "UNCHECKED") 
            checked = UNCHECKED;
        else if (tokens[x + 1] == "NOTCHECKABLE")
            checked = NOTCHECKABLE;
        else
        {
            VERBOSE(VB_ALL, "LCDServer: bad checked state in SWITCH_TO_MENU " 
                            "command: " << tokens[x + 1]);
            sendMessage(socket, "HUH?");
            return;
        }
        
        bool selected;
        if (tokens[x + 2] == "TRUE")
            selected = true;
        else if (tokens[x + 2] == "FALSE")
            selected = false; 
        else
        {
            VERBOSE(VB_ALL, "LCDServer: bad selected state in SWITCH_TO_MENU " 
                            "command: " << tokens[x + 2]);
            sendMessage(socket, "HUH?");
            return;
        }

        bool scrollable;
        if (tokens[x + 3] == "TRUE")
            scrollable = true;
        else if (tokens[x + 3] == "FALSE")
            scrollable = false; 
        else
        {
            VERBOSE(VB_ALL, "LCDServer: bad scrollable bool in SWITCH_TO_MENU " 
                            "command: " << tokens[x + 3]);
            sendMessage(socket, "HUH?");
            return;
        }
        
        bool bOK;
        int indent = tokens[x + 4].toInt(&bOK);
        if (!bOK)
        {
            VERBOSE(VB_ALL, "LCDServer: bad indent in SWITCH_TO_MENU " 
                            "command: " << tokens[x + 4]);
            sendMessage(socket, "HUH?");
            return;
        }

        items.append(new LCDMenuItem(selected, checked, text, indent));
    }
    
    if (m_lcd)
        m_lcd->switchToMenu(&items, appName, bPopup);
        
    sendMessage(socket, "OK");
}

void LCDServer::setChannelProgress(const QStringList &tokens, QSocket *socket)
{
    if (LCDSERVER_DEBUG)
        VERBOSE(VB_ALL, "LCDServer: SET_CHANNEL_PROGRESS");

    QString flat = tokens.join(" ");
   
    if (tokens.count() != 2)
    {
        VERBOSE(VB_ALL, "LCDServer: bad SET_CHANNEL_PROGRESS command: "
                << flat);
        sendMessage(socket, "HUH?");
        return;
    }
     
    bool bOK;
    float progress = tokens[1].toFloat(&bOK);
    if (!bOK)
    {
        VERBOSE(VB_ALL, "LCDServer: bad float value in SET_CHANNEL_PROGRESS " 
                        "command: %1" << tokens[1]);
        sendMessage(socket, "HUH?");
        return;
    }
    
    if (m_lcd)
        m_lcd->setChannelProgress(progress);
        
    sendMessage(socket, "OK");
}

void LCDServer::setMusicProgress(const QStringList &tokens, QSocket *socket)
{
    if (LCDSERVER_DEBUG)
        VERBOSE(VB_ALL, "LCDServer: SET_MUSIC_PROGRESS");

    QString flat = tokens.join(" ");
   
    if (tokens.count() != 3)
    {
        VERBOSE(VB_ALL, "LCDServer: bad SET_MUSIC_PROGRESS command: " << flat);
        sendMessage(socket, "HUH?");
        return;
    }
     
    bool bOK;
    float progress = tokens[2].toFloat(&bOK);
    if (!bOK)
    {
        VERBOSE(VB_ALL, "LCDServer: bad float value in SET_MUSIC_PROGRESS " 
                        "command: " << tokens[1]);
        sendMessage(socket, "HUH?");
        return;
    }
    
    if (m_lcd)
        m_lcd->setMusicProgress(tokens[1], progress);
        
    sendMessage(socket, "OK");
}

void LCDServer::setVolumeLevel(const QStringList &tokens, QSocket *socket)
{
    if (LCDSERVER_DEBUG)
        VERBOSE(VB_ALL, "LCDServer: SET_VOLUME_LEVEL");

    QString flat = tokens.join(" ");
   
    if (tokens.count() != 2)
    {
        VERBOSE(VB_ALL, "LCDServer: bad SET_VOLUME_LEVEL command: " << flat);
        sendMessage(socket, "HUH?");
        return;
    }
     
    bool bOK;
    float progress = tokens[1].toFloat(&bOK);
    if (!bOK)
    {
        VERBOSE(VB_ALL, "LCDServer: bad float value in SET_VOLUME_LEVEL " 
                        "command: " << tokens[1]);
        sendMessage(socket, "HUH?");
        return;
    }
    
    if (m_lcd)
        m_lcd->setVolumeLevel(progress);
        
    sendMessage(socket, "OK");
}

void LCDServer::updateLEDs(const QStringList &tokens, QSocket *socket)
{
    if (LCDSERVER_DEBUG)
        VERBOSE(VB_ALL, "LCDServer: UPDATE_LEDS");
    
    QString flat = tokens.join(" ");
   
    if (tokens.count() != 2)
    {
        VERBOSE(VB_ALL, "LCDServer: bad UPDATE_LEDs command: " << flat);
        sendMessage(socket, "HUH?");
        return;
    }

    bool bOK;
    int mask = tokens[1].toInt(&bOK);
    if (!bOK)
    {
        VERBOSE(VB_ALL, "LCDServer: bad mask in UPDATE_LEDS " 
                        "command: " << tokens[1]);
        sendMessage(socket, "HUH?");
        return;
    }
     
    if (m_lcd)
        m_lcd->updateLEDs(mask);
        
    sendMessage(socket, "OK");
}
