#ifndef _TELETEXTVIEW_H_
#define _TELETEXTVIEW_H_

// C includes
#include <stdint.h>

// C++ includes
#include <map>
using namespace std;

// Qt includes
#include <qmutex.h>
#include <qstring.h>

class TeletextDecoder;

class TeletextPage
{
  public:
    int           pagenum;      ///< the wanted page number
    int           subpagenum;
    unsigned char charset;

    int           lang;         ///< language code
    int           flags;        ///< misc flags (see PG_xxx below)
    int           errors;       ///< number of single bit errors in page
    long          lines;        ///< 1 bit for each line received
    unsigned char data[25][40]; ///< page contents
    int           flof;         ///< page has FastText links
    int           floflink[6];  ///< FastText links (FLOF)
    bool          subtitle;     ///< page is subtitle page
};
typedef map<int, TeletextPage> page_map_t;

class TeletextMagazine
{
	public:
		QMutex        lock;
		TeletextPage *current_page;
		page_map_t    pages;
};

class TTColor
{
	public:
		static const uint BLACK       = 0;
		static const uint RED         = 1;
		static const uint GREEN       = 2;
		static const uint YELLOW      = 3;
		static const uint BLUE        = 4;
		static const uint MAGNETA     = 5;
		static const uint CYAN        = 6;
		static const uint WHITE       = 7;
		static const uint TRANSPARENT = 8;
};

class TTKey
{
	public:
		static const uint k0           = 0;
		static const uint k1           = 1;
		static const uint k2           = 2;
		static const uint k3           = 3;
		static const uint k4           = 4;
		static const uint k5           = 5;
		static const uint k6           = 6;
		static const uint k7           = 7;
		static const uint k8           = 8;
		static const uint k9           = 9;
		static const uint kNextPage    = 10;
		static const uint kPrevPage    = 11;
		static const uint kNextSubPage = 12;
		static const uint kPrevSubPage = 13;
		static const uint kHold        = 14;
		static const uint kTransparent = 15;
		static const uint kFlofRed     = 16;
		static const uint kFlofGreen   = 17;
		static const uint kFlofYellow  = 18;
		static const uint kFlofBlue    = 19;
};

class TeletextView
{
	public:
		TeletextView();
		virtual ~TeletextView();

		void SetDecoder(TeletextDecoder*);

		virtual void SetBackgroundColor(int color) = 0;
		virtual void SetForegroundColor(int color) = 0;

		int GetCurrentPage(void) const { return m_curpage; };
		bool IsVisible(void)     const { return m_visible; };

		virtual void DrawBackground(int x, int y) = 0;
		virtual void DrawCharacter(int x, int y, QChar ch, int doubleheight) = 0;
		virtual void DrawMosaic(int x, int y, int code, int doubleheight) = 0;

		virtual void NewsFlash(void) = 0;

		virtual void PageUpdated(int page) = 0;
		virtual void HeaderUpdated(uint8_t* page, int lang) = 0;

		virtual void KeyPress(uint key);
    void DecoderDestroyed(void) { /*LOCK*/ m_decoder = NULL; };

  protected:
    void DrawLine(const uint8_t* page, int row, int lang);
    void DrawHeader(const uint8_t* page, int lang);
    void DrawPage(void);
    char CharConversion(char ch, int lang);
    
  protected:
    TeletextDecoder    *m_decoder;

    char                m_pageinput[3];
    int                 m_curpage;
    int                 m_cursubpage;
    bool                m_curpage_showheader;
    
    bool                m_pageupdated;
    uint8_t            *m_curheader;
    int                 m_curheaderlang;

    int                 m_sidebar;
    
    bool                m_visible;
    bool                m_transparent;
    bool                m_header;

  public:
    static const uint COLS = 40;
    static const uint ROWS = 25;
};

#endif // _TELETEXTVIEW_H_
