/* =============================================================
 * File  : osdtypeteletext.cpp
 * Author: Frank Muenchow <beebof@gmx.de>
 *         Martin Barnasconi
 * Date  : 2005-10-25
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * ============================================================= */
#include "osd.h"
#include "osdsurface.h"
#include "osdtypes.h"
#include "osdtypeteletext.h"
#include "ttfont.h"
#include "vbilut.h"

#include <qcolor.h>
#include <qapplication.h>
#include <iostream>
#include <cmath>
#include <qmutex.h>

using namespace std;

QColor m_color_black;
QColor m_color_red;
QColor m_color_green;
QColor m_color_yellow;
QColor m_color_blue;
QColor m_color_magenta;
QColor m_color_cyan;
QColor m_color_white;

/*****************************************************************************/
OSDTypeTeletext::OSDTypeTeletext(const QString &name, QRect displayrect,
                                 TTFFont *font)
    : OSDType(name)
{
    m_surface = NULL;

    m_displayrect = displayrect;
    m_font = font;

    m_tt_colspace = m_displayrect.width()  / COLS;
    m_tt_rowspace = m_displayrect.height() / ROWS;

    m_color_black   = QColor(  0,  0,  0);
    m_color_red     = QColor(255,  0,  0);
    m_color_green   = QColor(  0,255,  0);
    m_color_yellow  = QColor(255,255,  0);
    m_color_blue    = QColor(  0,  0,255);
    m_color_magenta = QColor(255,  0,255);
    m_color_cyan    = QColor(  0,255,255);
    m_color_white   = QColor(255,255,255);

    for (int i = 0; i < 8; ++i)
    {
        QMutexLocker lock(&m_magazines[i].lock);

	m_magazines[i].pages.clear();
	m_magazines[i].current_page = NULL;
    }
    m_currentpage = 0x100;

    // fill Bitswap
    for (int i = 0; i < 256; i++)
    {
        bitswap[i] = 0;
	for (int bit = 0; bit < 8; bit++)
            if (i & (1 << bit))
                bitswap[i] |= (1 << (7-bit));
    }
}

OSDTypeTeletext::OSDTypeTeletext(const OSDTypeTeletext &other)
    : OSDType(other.m_name)
{
}

OSDTypeTeletext::~OSDTypeTeletext()
{
}

/** \fn OSDTypeTeletext::CharConversion(char, int)
 *  \brief converts the given character to the given language
 */
char OSDTypeTeletext::CharConversion(char ch, int lang)
{
    int c = 0;
    for (int j = 0; j < 14; j++)
    {
        c = ch & 0x7F;
	if (c == lang_chars[0][j])
	    ch = lang_chars[lang + 1][j];
    }
    return ch;
}

/** \fn OSDTypeTeletext::AddPageHeader(int, int, uint8_t*, int)
 *  \brief Adds a new page header 
 *         (page=0 if none)
 */
void OSDTypeTeletext::AddPageHeader(int page, int subpage, const uint8_t* buf, int vbimode)
{
    PageUpdated(m_currentpage);

    m_currentpage = page;
    m_currentsubpage = subpage;
   
    if (page < 256)
        return;

    TeletextPage *ttpage = FindPage(page, subpage);
    if (ttpage == NULL)
    {
	int magazine = m_currentpage / 256;
        ttpage = &m_magazines[magazine - 1].pages[m_currentsubpage * 256 + (m_currentpage % 256)];
	ttpage->pagenum = page;
	ttpage->subpagenum = subpage;
    } 

    ttpage->lang = 0; // will be set in SetLanguage!
    ttpage->flags = 0; // will be set in SetFlags!
    ttpage->flof = 0;
    
    for (int i=0; i < 6; i++)
	ttpage->floflink[i] = 0;

    ttpage->subtitle = (vbimode == VBI_DVB_SUBTITLE);

    for (int j = 0; j < 8; j++)
	ttpage->data[0][j] = 32;

    if (vbimode == VBI_DVB || vbimode == VBI_DVB_SUBTITLE)
    {
	for (int j = 8; j < 40; j++)
            ttpage->data[0][j] = bitswap[buf[j]];
    } else {
	memcpy(ttpage->data[0]+0, buf, 40);
	memset(ttpage->data[0]+40, ' ', sizeof(ttpage->data)-40);
    }

    if ((ttpage->flags & 0xC1) || (vbimode == VBI_DVB_SUBTITLE))
	for (int j = 1; j < 24; j++)
	    ttpage->data[j][0] = 255;
    
    HeaderUpdated(ttpage->data[0],ttpage->lang);
}

/** \fn OSDTypeTeletext::AddTeletextData(int, int, uint8_t*)
 *  \brief Adds Teletext Data from TeletextDecoder
 */
void OSDTypeTeletext::AddTeletextData(int magazine, int row, const uint8_t* buf, int vbimode)
{
    int b1, b2, b3, err;

    if (magazine != (m_currentpage / 256))
        return;
    if (m_currentpage == 0)
	return;

    TeletextPage *ttpage = FindPage(m_currentpage, m_currentsubpage);

    if (ttpage == NULL)
	return;

    switch (row)
    {
	case 1 ... 24: // Page Data
	    if (vbimode == VBI_DVB || vbimode == VBI_DVB_SUBTITLE)
	    {
		for (int j = 0; j < 40; j++)
		    ttpage->data[row][j] = bitswap[buf[j]];
	    } else {
		memcpy(ttpage->data[row], buf, 40);
	    }
	    break;
	case 26:
	    /* XXX TODO: Level 1.5, 2.5, 3.5
	     *      Character location & override
	     * Level 2.5, 3.5
	     *      Modifying display attributes
	     * All levels
	     *      VCR Programming
	     * See 12.3
	     */
	    break;
        case 27: // FLOF data (FastText)
	    switch (vbimode)
	    {
		case VBI_IVTV:
		    b1 = hamm8(buf, &err);
		    b2 = hamm8(buf + 37, & err);
		    if (err & 0xF00)
		        return;
		    break;
		case VBI_DVB:
		case VBI_DVB_SUBTITLE:
		    b1 = hamm84(buf, &err);
		    b2 = hamm84(buf + 37, &err);
		    if (err == 1)
			return;
		    break;
		default:
		    return;
	    }
	    if (b1 != 0 || not(b2 & 8))
		return;

	    for (int i = 0; i < 6; ++i)
	    {
		err = 0;
		switch (vbimode)
		{
	            case VBI_IVTV:
			b1 = hamm16(buf+1+6*i, &err);
			b2 = hamm16(buf+3+6*i, &err);
			b3 = hamm16(buf+5+6*i, &err);
			if (err & 0xF000)
		            return;
			break;
	            case VBI_DVB:
		    case VBI_DVB_SUBTITLE:
			b1 = hamm84(buf+2+6*i, &err) * 16 +
			   hamm84(buf+1+6*i, &err);
			b2 = hamm84(buf+4+6*i, &err) * 16 +
			   hamm84(buf+3+6*i, &err);
			b3 = hamm84(buf+6+6*i, &err) * 16 +
			   hamm84(buf+5+6*i, &err);
			if (err == 1)
			    return;
			break;
		    default:
			return;
		}
		int x = (b2 >> 7) | ((b3 >> 5) & 0x06);
		ttpage->floflink[i] = ((magazine ^ x) ?: 8) * 256 + b1;
		ttpage->flof = 1;
	    }
	    break;
	case 31: // private streams
	    break;
	default: /// other packet codes...
	    break;
    }
}

/** \fn OSDTypeTeletext::SetLanguage(int)
 *  \brief Sets the language of the current page
 */
void OSDTypeTeletext::SetLanguage(int lang)
{
    TeletextPage *ttpage = FindPage(m_currentpage,m_currentsubpage);
    
    if (ttpage == NULL)
        return;
    
    ttpage->lang = lang;
}

/** \fn OSDTypeTeletext::SetFlags(int)
 *  \brief Sets some flags of the current page
 */
void OSDTypeTeletext::SetFlags(int flags)
{
    TeletextPage *ttpage = FindPage(m_currentpage, m_currentsubpage);

    if (ttpage == NULL)
        return;

    ttpage->flags = flags;
}

/** \fn OSDTypeTeletext::PageUpdated(int)
 *  \brief Updates the page, if given pagenumber is the same
 *         as the current shown page
 */
void OSDTypeTeletext::PageUpdated(int page)
{
    return; 
#warning FIXME - PageUpdated

    if (page != m_curpage)
	return;
    
    m_surface->SetChanged(true);
    m_surface->ClearUsed();
    DrawPage();
}

/** \fn OSDTypeTeletext::HeaderUpdated(uint8_t*,int)
 *  \brief Updates the header (given in page with language lang)
 * 
 *  \param page Pointer to the header which should be displayed
 *  \param lang Language of the header
 *
 */
void OSDTypeTeletext::HeaderUpdated(uint8_t *page, int lang)
{
    return;
#warning FIXME - HeaderUpdated
  
    if (page == NULL)
        return;

    if (m_curpage_showheader == false)
	return;
   
    DrawHeader(page,lang);
}

/** \fn OSDTypeTeletext::FindPage(int, int)
 *  \brief Finds the given page
 *         Returns "true" if page not found
 *
 *  \param page Page number
 *  \param subpage Subpage number (if set to -1, find the first subpage)
 *  \return TeletextPage (NULL if not found) 
 */
TeletextPage* OSDTypeTeletext::FindPage(int page, int subpage)
{
    int mag = page / 256;
    if (mag > 8 || mag < 1)
        return NULL;

    QMutexLocker lock(&m_magazines[mag - 1].lock);

    if (subpage == -1)
    {
	// Search the first 10 subpages
	// and return the first found
	std::map<int, TeletextPage>::iterator iter;
	int j = 0;
        do {
            iter = m_magazines[mag - 1].pages.find(j * 256 + page % 256);
            j++;
	} while ((iter == m_magazines[mag - 1].pages.end()) && (j < 10));
	
	if (iter == m_magazines[mag - 1].pages.end())
            return NULL;

	return &iter->second;
    }
    else
    {
        std::map<int, TeletextPage>::iterator iter;
	iter = m_magazines[mag - 1].pages.find(subpage * 256 + page % 256);
	if (iter == m_magazines[mag - 1].pages.end())
	    return NULL;

	return &iter->second;
    }
}

/** \fn OSDTypeTeletext::KeyPress(uint)
 *  \brief What to do if a key is pressed
 *
 *  \param key pressed key
 *  \sa TTKey
 */
void OSDTypeTeletext::KeyPress(uint key)
{
    bool numeric_input = false;
    TeletextPage *curpage = FindPage(m_curpage, m_cursubpage);
    TeletextPage *page;

    switch (key)
    {
        case TTKey::k0 ... TTKey::k9:
            numeric_input = true;
	    m_curpage_showheader = true;
	    if (m_pageinput[0] == ' ')
	        m_pageinput[0] = '0' + static_cast<int> (key);
	    else if (m_pageinput[1] == ' ')
		m_pageinput[1] = '0' + static_cast<int> (key);
	    else if (m_pageinput[2] == ' ')
	    {
		m_pageinput[2] = '0' + static_cast<int> (key);
		m_curpage = ((m_pageinput[0] - '0') * 256) +
		    ((m_pageinput[1] - '0') * 16) +
		    (m_pageinput[2] - '0');
		m_cursubpage = -1;
	    }
	    else
	    {
	        m_pageinput[0] = '0' + static_cast<int> (key);
		m_pageinput[1] = ' ';
		m_pageinput[2] = ' ';
	    }
	    break;
        case TTKey::kNextPage:
	    m_curpage++;
	    m_cursubpage = -1;
	    m_curpage_showheader = true;
	    if (m_curpage % 16 == 10)
		m_curpage += 6;
	    if (m_curpage % 256 == 160)
		m_curpage += 96;
	    break;
       case TTKey::kPrevPage:
	    m_curpage--;
	    m_cursubpage = -1;
	    m_curpage_showheader = true;
	    if (m_curpage % 16 == 10)
		m_curpage += 6;
	    if (m_curpage % 256 == 160)
		m_curpage += 96;
	    break;
	case TTKey::kNextSubPage:
	    m_cursubpage++;
	    m_curpage_showheader = true;
	    break;
	case TTKey::kPrevSubPage:
	    m_cursubpage--;
	    if (m_cursubpage == -1)
		m_cursubpage = 0;
	    m_curpage_showheader = true;
	    break;
	case TTKey::kHold:
	    break;
	case TTKey::kTransparent:
	    m_transparent = !m_transparent;
	    break;
	case TTKey::kFlofRed:
	    if ((page = FindPage(curpage->floflink[0], -1)) != NULL)
            {
		m_curpage = page->pagenum;
		m_cursubpage = -1;
		m_curpage_showheader = true;
	    }
	    break;
	case TTKey::kFlofGreen:
	    if ((page = FindPage(curpage->floflink[1], -1)) != NULL)
	    {
		m_curpage = page->pagenum;
		m_cursubpage = -1;
		m_curpage_showheader = true;
            }
	    break;
	case TTKey::kFlofYellow:
	    if ((page = FindPage(curpage->floflink[2], -1)) != NULL)
	    {
		m_curpage = page->pagenum;
		m_cursubpage = -1;
		m_curpage_showheader = true;
	    }
	case TTKey::kFlofBlue:
	    if ((page = FindPage(curpage->floflink[3], -1)) != NULL)
	    {
		m_curpage = page->pagenum;
		m_cursubpage = -1;
		m_curpage_showheader = true;
	    }
	    break;
    }

    if (m_curpage < 0x100)
	m_curpage = 0x100;
    if (m_curpage > 0x899)
	m_curpage = 0x899;

    if (!numeric_input)
    {
	m_pageinput[0] = (m_curpage / 256) + '0';
	m_pageinput[1] = ((m_curpage % 256) / 16) + '0';
	m_pageinput[2] = (m_curpage % 16) + '0';
    }
    PageUpdated(m_curpage);
}

/** \fn OSDTypeTeletext::SetForegroundColor(int)
 *  \brief Set the font color to the given color.
 *
 *   NOTE: TTColor::TRANSPARENT does not do anything!
 *  \sa TTColor
 */
void OSDTypeTeletext::SetForegroundColor(int color)
{
    switch (color & ~TTColor::TRANSPARENT)
    {
        case TTColor::BLACK:   m_font->setColor(m_color_black);   break;
        case TTColor::RED:     m_font->setColor(m_color_red);     break;
        case TTColor::GREEN:   m_font->setColor(m_color_green);   break;
        case TTColor::YELLOW:  m_font->setColor(m_color_yellow);  break;
        case TTColor::BLUE:    m_font->setColor(m_color_blue);    break;
        case TTColor::MAGENTA: m_font->setColor(m_color_magenta); break;
        case TTColor::CYAN:    m_font->setColor(m_color_cyan);    break;
        case TTColor::WHITE:   m_font->setColor(m_color_white);   break;
    }

    m_font->setShadow(0,0);
    m_font->setOutline(0);
}

/** \fn OSDTypeTeletext:SetBackgroundColor(int)
 *  \brief Set the background color to the given color
 *
 *   NOTE: TTColor::TRANSPARENT does not do anything!
 *  \sa TTColor
 */
void OSDTypeTeletext::SetBackgroundColor(int ttcolor)
{
    QColor color;

    switch (ttcolor & ~TTColor::TRANSPARENT)
    {
        case TTColor::BLACK:   color = m_color_black;   break;
        case TTColor::RED:     color = m_color_red;     break;
        case TTColor::GREEN:   color = m_color_green;   break;
        case TTColor::YELLOW:  color = m_color_yellow;  break;
        case TTColor::BLUE:    color = m_color_blue;    break;
        case TTColor::MAGENTA: color = m_color_magenta; break;
        case TTColor::CYAN:    color = m_color_cyan;    break;
        case TTColor::WHITE:   color = m_color_white;   break;
    }

    int r = color.red();
    int g = color.green();
    int b = color.blue();

    float y = (0.299*r) + (0.587*g) + (0.114*b);
    float u = (0.564*(b - y)); // = -0.169R-0.331G+0.500B
    float v = (0.713*(r - y)); // = 0.500R-0.419G-0.081B

    m_bgcolor_y = (uint8_t)(y);
    m_bgcolor_u = (uint8_t)(127 + u);
    m_bgcolor_v = (uint8_t)(127 + v);
    m_bgcolor_a = 0xff;
}

/** \fn OSDTypeTeletext::DrawBackground(int, int)
 *  \brief draws background in the color set in setBackgroundColor
 *
 *  \param x X position (40 cols)
 *  \param y Y position (25 rows)
 */
void OSDTypeTeletext::DrawBackground(int x, int y)
{
    x *= m_tt_colspace;
    x += m_displayrect.left();

    y *= m_tt_rowspace;
    y += m_displayrect.top();

    DrawRect(x, y, m_tt_colspace, m_tt_rowspace);
}

/** \fn OSDTypeteletext::drawRect(int, int, int, int)
 *  \brief draws a Rectangle at position x and y (width dx, height dy)
 *         with the color set in setBackgroundColor
 *
 *  \param x X position at the screen
 *  \param y Y position at the screen
 *  \param dx width
 *  \param dy height
 */
void OSDTypeTeletext::DrawRect(int x, int y, int dx, int dy)
{
    QRect all(x, y, dx, dy);
    m_surface->AddRect(all);
    
    int luma_stride = m_surface->width;
    int chroma_stride = m_surface->width >> 1;
    int ye = y + dy;
    
    unsigned char *buf_y = m_surface->y + (luma_stride * y) + x;
    unsigned char *buf_u = m_surface->u + (chroma_stride * (y>>1)) + (x>>1);
    unsigned char *buf_v = m_surface->v + (chroma_stride * (y>>1)) + (x>>1);
    unsigned char *buf_a = m_surface->alpha + (luma_stride * y) + x;

    for (; y<ye; ++y)
    {
        for (int i=0; i<dx; ++i)
        {
            buf_y[i] = m_bgcolor_y;
            buf_a[i] = m_bgcolor_a;
        }

        if ((y & 1) == 0)
        {
            for (int i=0; i<dx; ++i)
            {
                buf_u[i>>1] = m_bgcolor_u;
                buf_v[i>>1] = m_bgcolor_v;
            }

            buf_u += chroma_stride;
            buf_v += chroma_stride;
        }

        buf_y += luma_stride;
        buf_a += luma_stride;
    }
}

/** \fn OSDTypeTeletext::DrawCharacter(int, int, QChar, int)
 *  \brief Draws a character at posistion x, y 
 *
 *  \param x X position (40 cols)
 *  \param y Y position (25 rows)
 *  \param ch Character
 *  \param doubleheight if different to 0, draw doubleheighted character
 */
void OSDTypeTeletext::DrawCharacter(int x, int y, QChar ch, int doubleheight)
{
    if (!m_font)
        return;

    QString line = ch;

    x *= m_tt_colspace;
    x += m_displayrect.left();

    y *= m_tt_rowspace;
    y += m_displayrect.top();

    m_font->DrawString(m_surface, x, y, line,
                       m_surface->width, m_surface->height,
                       255, doubleheight!=0);
}

/** \fn OSDTypeTeletext::DrawMosaic(int, int, code, int)
 *  \brief Draws a mosaic as defined in ETSI EN 300 706
 *
 *  \param x X position (40 cols)
 *  \param y Y position (25 rows)
 *  \param code Code
 *  \param doubleheight if different to 0, draw doubleheighted mosaic)
 */
void OSDTypeTeletext::DrawMosaic(int x, int y, int code, int doubleheight)
{
    (void)x;
    (void)y;
    (void)code;

    x *= m_tt_colspace;
    x += m_displayrect.left();

    y *= m_tt_rowspace;
    y += m_displayrect.top();
    
    int dx = (int)round(m_tt_colspace / 2)+1;
    int dy = (int)round(m_tt_rowspace / 3)+1;
   
    if (doubleheight != 0)
    {
	dx *= 2;
	dy *= 2;
    }

    if (code & 0x10) DrawRect(x,      y + 2*dy, dx, dy);
    if (code & 0x40) DrawRect(x + dx, y + 2*dy, dx, dy);
    if (code & 0x01) DrawRect(x,      y,        dx, dy);
    if (code & 0x02) DrawRect(x + dx, y,        dx, dy);
    if (code & 0x04) DrawRect(x,      y + dy,   dx, dy);
    if (code & 0x08) DrawRect(x + dx, y + dy,   dx, dy);
}

void OSDTypeTeletext::DrawLine(const uint8_t* page, uint row, int lang)
{
    bool mosaic;
    bool conceal;
    bool seperation;
    bool flash;
    bool doubleheight;
    bool blink;
    bool hold;

    char last_ch = ' ';
    char ch;

    uint fgcolor = TTColor::WHITE;
    uint bgcolor = TTColor::BLACK;
    uint newfgcolor = TTColor::WHITE;
    uint newbgcolor = TTColor::BLACK;

    SetForegroundColor(fgcolor);
    SetBackgroundColor(bgcolor);

    mosaic = false;
    seperation = false;
    conceal = false;
    flash = false;
    doubleheight = false;
    blink = false;
    hold = false;
    
    for (unsigned int  x = 0; x < COLS; ++x)
    {
        SetForegroundColor(fgcolor);
	SetBackgroundColor(bgcolor);

	ch = page[x] & 0x7F;
	switch (ch)
	{
            case 0x00 ... 0x07: // alpha + foreground color
                fgcolor = ch & 7;
		mosaic = false;
		conceal = false;
		goto ctrl;
            case 0x08: // flash
		// XXX
		goto ctrl;
            case 0x09: // steady
		flash = false;
		goto ctrl;
	    case 0x0a: // end box
	    case 0x0b: // start box
		goto ctrl;
	    case 0x0c: // normal height
		doubleheight = 0;
		goto ctrl;
	    case 0x0d: // double height
		doubleheight = row < ROWS-1;
		goto ctrl;
	    case 0x10 ... 0x17: // graphics + foreground color
		fgcolor = ch & 7;
		mosaic = true;
		conceal = false;
		goto ctrl;
	    case 0x18: // conceal display
		conceal = false;
		goto ctrl;
	    case 0x19: // contiguous graphics
		seperation = false;
		goto ctrl;
	    case 0x1a: // separate graphics
		seperation = true;
		goto ctrl;
	    case 0x1c: // black background
		bgcolor = TTColor::BLACK;
		goto ctrl;
	    case 0x1d: // new background
		bgcolor = fgcolor;
		goto ctrl;
	    case 0x1e: // hold graphics
		hold = true;
		goto ctrl;
	    case 0x1f: // release graphics
		hold = false;
		goto ctrl;
	    case 0x0e: // SO (reserved, double width)
	    case 0x0f: // SI (reserved, double size)
	    case 0x1b: // ESC (reserved)
		ch = ' ';
		break;
	    ctrl:
		ch = ' ';
		if (hold && mosaic)
		    ch = last_ch;
		break;

	    case 0x80 ... 0x9f: // these aren't used
		ch = ' '; // BAD_CHAR;
		break;
	    default:
	        break;
	}

	newfgcolor = fgcolor;
	newbgcolor = bgcolor;

	SetForegroundColor(newfgcolor);
	SetBackgroundColor(newbgcolor);
	if ((row != 0) || (x > 7))
	{
	    if (!m_transparent)
	    {
		DrawBackground(x, row);
		if (doubleheight != 0)
	            DrawBackground(x, row +1);
	    }
	    if ((mosaic) && (ch < 0x40 || ch > 0x5F))
	    {
		SetBackgroundColor(newfgcolor);
		DrawMosaic(x, row, ch, doubleheight);
	    } else {
		if (doubleheight != 0)
		    DrawCharacter(x, row+1, CharConversion(ch, lang), doubleheight);
		else
		    DrawCharacter(x, row, CharConversion(ch, lang), doubleheight);
	    }
	}
    }
}

void OSDTypeTeletext::DrawHeader(const uint8_t* page, int lang)
{
    DrawLine(page, 0, lang);

    SetForegroundColor(TTColor::WHITE);
    SetBackgroundColor(TTColor::BLACK);

    for (int i = 0; i < 8; ++i)
	DrawBackground(i, 0);

    DrawCharacter(1, 0, m_pageinput[0], 0);
    DrawCharacter(2, 0, m_pageinput[1], 0);
    DrawCharacter(3, 0, m_pageinput[2], 0);
    if (m_cursubpage != -1)
    {
	DrawCharacter(4, 0, QChar('/'), 0);
	DrawCharacter(5, 0, m_cursubpage + '0', 0);
    }
}

void OSDTypeTeletext::DrawPage(void)
{
    TeletextPage *ttpage = FindPage(m_curpage, m_cursubpage);
    
    if (ttpage == NULL)
    {
	SetBackgroundColor(TTColor::BLACK);
	SetForegroundColor(TTColor::RED);

	if (!m_transparent)
	    for (int i = 1; i < 40; i++)
		DrawBackground(i, 1);

	char *str = "Teletext page unavailable";
	for (unsigned int i = 0; i < strlen(str); ++i)
            DrawCharacter(i+7, 1, str[i], 0);
	
	return;
    }
    
    m_cursubpage = ttpage->subpagenum;

    int a = 0;

    if ((ttpage->subtitle) || (ttpage->flags & 0xC1))
        a = 1;
    else
        DrawHeader(ttpage->data[0], ttpage->lang);

    for (int y = ROWS-1-a; y >= 1; y--)
	DrawLine(ttpage->data[y], y, ttpage->lang);

}

void OSDTypeTeletext::Reinit(float wchange, float hchange)
{
    m_displayrect = QRect((int)(m_displayrect.x() * wchange),
                          (int)(m_displayrect.y() * hchange),
                          (int)(m_displayrect.width() * wchange),
                          (int)(m_displayrect.height() * hchange));

    m_tt_colspace = m_displayrect.width()  / COLS;
    m_tt_rowspace = m_displayrect.height() / ROWS;
}

void OSDTypeTeletext::Draw(OSDSurface *surface, int fade, int maxfade,
                           int xoff, int yoff)
{
    (void)surface;
    (void)fade;
    (void)maxfade;
    (void)xoff;
    (void)yoff;

    m_surface = surface;
    DrawPage();
}
