#include <stdio.h>
#include <string.h>
#include <stdint.h>
#include <ctype.h>

extern "C" {
#include <inttypes.h>
#include "ivtv_myth.h"
}

using namespace std;

#include "vbidecoder.h"
#include "vbilut.h"
#include "mythcontext.h"

//
// Teletext section
//

#define printable(c) ((((c) & 0x7F) < 0x20 || ((c) & 0x7F) > 0x7E) ? '.' : ((c) & 0x7F))

#define VBI_STARTBOX 0x0b
#define VBI_ENDBOX 0x0a
#define VBI_SEPARATEDMOSIACS 0x1a
#define VBI_HOLDMOSIACS 0x1e
#define VBI_RELEASEMOSIACS 0x1f
#define VBI_RELEASEMOSIACS 0x1f

TeletextPage *m_serialpage = NULL;

TeletextView::TeletextView()
{
    m_decoder = NULL;

    m_pageinput[0] = '1';
    m_pageinput[1] = '0';
    m_pageinput[2] = '0';

    m_curpage = 0x100;
    m_cursubpage = -1; // -1: select first available subpage
    m_curpage_showheader = true;
    m_pageupdated = true;

    m_sidebar = 0;
    m_visible = false;
    m_transparent = false;

    m_tt_rows = 25;
    m_tt_cols = 40;

    
}

TeletextView::~TeletextView()
{
}

void TeletextView::setDecoder(class VbiDecoder *dec)
{
    // XXX - Lock?
    m_decoder = dec;

    connect(dec, SIGNAL(pageUpdated(int)), this, SLOT(pageUpdated(int)));
    connect(dec, SIGNAL(headerUpdated(uint8_t*,int)), this, SLOT(headerUpdated(uint8_t*,int)));
}

void TeletextView::setSideBar(int div)
{
    // XXX
    m_sidebar = div;
}

void TeletextView::drawLine(uint8_t* page, int row, int lang)
{
    bool mosaic;
    bool conceal;
    bool seperation;
    bool flash;
    bool doubleheight;
    bool blink;
    bool hold;
    
    char last_ch = ' ';
    char ch;
    
    int fgcolor = TTCOLOR_WHITE;
    int newfgcolor = TTCOLOR_WHITE;
    int bgcolor = TTCOLOR_BLACK;
    int newbgcolor = TTCOLOR_BLACK;
   
    if (page[0]!=255)
    {
        fgcolor = TTCOLOR_WHITE;
        newfgcolor = TTCOLOR_WHITE;
	bgcolor = TTCOLOR_BLACK;

	setForegroundColor(fgcolor);
	setBackgroundColor(bgcolor);
	
        mosaic = false;
        seperation = false;
        conceal = false;
        flash = false;
        doubleheight = false;
        blink = false;
        hold = false;
        seperation = false;
        
        for (int x = 0; x < m_tt_cols; ++x)
        {
	    setForegroundColor(fgcolor);
	    setBackgroundColor(bgcolor);
	   
	    ch = page[x] & 0x7F;
	    switch(ch) {
                case 0x00 ... 0x07:	// alpha + foreground color 
                    fgcolor = ch & 7;
                    mosaic = false;
                    conceal = false;
                    goto ctrl;
                case 0x08:		// flash 
                    //flash_state = not xw->blink_on;
                    goto ctrl;
                case 0x09:		// steady 
                    flash = false;
                    goto ctrl;
                case 0x0a:		// end box 
                case 0x0b:		// start box 
                    goto ctrl;
                case 0x0c:		// normal height 
                    doubleheight = 0;
                    goto ctrl;
                case 0x0d:		// double height
                    doubleheight = row < m_tt_rows-1;
                    goto ctrl;
                case 0x10 ... 0x17:	// graphics + foreground color 
                    fgcolor = ch & 7;
                    mosaic = true;
                    conceal = false;
                    goto ctrl;
                case 0x18:		// conceal display 
		    conceal = false; //not xw->reveal;
                    goto ctrl;
                case 0x19:		// contiguous graphics
                    seperation = false;
                    goto ctrl;
                case 0x1a:		// separate graphics 
                    seperation = true;
                    goto ctrl;
                case 0x1c:		// black background
                    bgcolor = TTCOLOR_BLACK;
                    goto ctrl;
                case 0x1d:		// new background
                    bgcolor = fgcolor;
                    goto ctrl;
                case 0x1e:		// hold graphics 
                    hold = true;
                    goto ctrl;
                case 0x1f:		// release graphics
                    hold = false;
                    goto ctrl;
                case 0x0e:		// SO (reserved, double width)
                case 0x0f:		// SI (reserved, double size)
                case 0x1b:		// ESC (reserved)
                    ch = ' ';
                    break;
		ctrl:
                    ch = ' ';
                    if (hold && mosaic)
                        ch = last_ch;
                    break;
                
		case 0x80 ... 0x9f:	// these aren't used
                    ch = ' ' ;//BAD_CHAR;
                    break;

                default:		// mapped to selected font
                    break;
            }
                
            newfgcolor = fgcolor;
            newbgcolor = bgcolor;

            if (blink)
            {
                //XXX TODO 
                //newfgcolor |= 8;                
            }                    

            //if (conceal)
                //newfgcolor = newbgcolor;

	    setForegroundColor(newfgcolor);
            setBackgroundColor(newbgcolor);
		
            if (!m_transparent) 
	    {
		drawBackground(x, row);
	        if (doubleheight != 0)  
		    drawBackground(x, row+1);
	    }

            if ((mosaic) && (ch < 0x40 || ch > 0x5F))
	    {
	        setBackgroundColor(newfgcolor);
	        drawMosaic(x, row, ch, doubleheight);
	    }
	    else
	    {
	        if (doubleheight != 0)
	            drawCharacter(x, row+1, charConversion(ch,lang), doubleheight);
		else
		    drawCharacter(x, row, charConversion(ch,lang), doubleheight);
	    }
	}
    }
}

void TeletextView::drawPage()
{
    TeletextPage *page;
    
    if (!m_decoder)
	return;

    bool found = m_decoder->findPage(m_curpage, m_cursubpage, &page);
   
    if (!m_pageupdated)
    {
	if (m_curpage_showheader)
            drawHeader(m_curheader,page->lang);
    }
    
    if (!found)
    {
	setBackgroundColor(TTCOLOR_BLACK);
        setForegroundColor(TTCOLOR_RED);
        
	if (!m_transparent)
            for (int i=1; i<40; ++i)
                drawBackground(i, 1);

        char *str = "Teletext page unavailable";
        for (unsigned int i=0; i<strlen(str); ++i)
        {
            drawCharacter(i+7, 1, str[i], 0);
        }
        return;
    }
   
    m_cursubpage = page->subpagenum;

    int a = 0;
    if ((page->subtitle) || (page->flags & 0xc1)) // if subtitle, don't display FLOF infos
    {
	m_curpage_showheader = false;
	a = 1;
    }
    else
    {
	m_curpage_showheader = true;
	drawHeader(page->data[0], page->lang);
    }

    for (int y = m_tt_rows-1-a; y >= 1; y--)
    {
	drawLine(page->data[y], y, page->lang);
    }
}

void TeletextView::drawHeader(uint8_t* page, int lang)
{
    drawLine(page, 0, lang);
    
    setForegroundColor(TTCOLOR_WHITE);
    setBackgroundColor(TTCOLOR_BLACK);
   
    drawCharacter(1, 0, m_pageinput[0], 0);
    drawCharacter(2, 0, m_pageinput[1], 0);
    drawCharacter(3, 0, m_pageinput[2], 0);
    if (m_cursubpage != -1)
    {
        drawCharacter(4, 0, QChar('/'), 0);
        //drawCharacter(5, 0, page->subpagenum + '0');
        drawCharacter(5, 0, m_cursubpage + '0', 0);
    }
}

void TeletextView::keyPress(enum TTKey key)
{
    bool numeric_input = false;

    TeletextPage *page;

    if(!m_decoder)
        return;

    switch (key)
    {
        case TTKey_0 ... TTKey_9:
            numeric_input = true;

            if (m_pageinput[0] == ' ')
                m_pageinput[0] = '0' + static_cast<int> (key);
            else if (m_pageinput[1] == ' ')
                m_pageinput[1] = '0' + static_cast<int> (key);
            else if (m_pageinput[2] == ' ')
            {
                m_pageinput[2] = '0' + static_cast<int> (key);
                // XXX - Signal pagechange.
                m_curpage = ((m_pageinput[0] - '0') * 256) +
                            ((m_pageinput[1] - '0') * 16) +
                            (m_pageinput[2] - '0');
		m_cursubpage = -1;
		m_curpage_showheader = true;
            }
            else
            {
                // XXX - Abort pagechange.
                m_pageinput[0] = '0' + static_cast<int> (key);
                m_pageinput[1] = ' ';
                m_pageinput[2] = ' ';
            }
            break;

        case TTKey_NextPage:
            m_curpage++;
	    m_cursubpage = -1;
	    m_curpage_showheader = true;
            if (m_curpage % 16 == 10)
		    m_curpage += 6;
	    if (m_curpage % 256 == 160)
		    m_curpage += 96;
	    break;
        case TTKey_PrevPage:
            m_curpage--;
	    m_cursubpage = -1;
	    m_curpage_showheader = true;
	    if (m_curpage % 16 == 15)
		    m_curpage -= 6;
	    if (m_curpage % 256 == 249)
		    m_curpage -= 96;
            break;
        case TTKey_NextSubPage:
            // m_curpage += 16;
	    // if (m_curpage % 256 > 159)
            //     m_curpage += 96;
	    m_cursubpage++; // TODO: select next *possible* subpage
	    m_curpage_showheader = true;
            break;
        case TTKey_PrevSubPage:
            // m_curpage -= 16;
	    // if (m_curpage % 256 > 159)
            //     m_curpage -=96;
	    m_cursubpage--; // TODO: select previous *possible* subpage
	    if (m_cursubpage == -1)
                m_cursubpage = 0;
	    m_curpage_showheader = true;
            break;

        case TTKey_Hold:
	    break;
        case TTKey_Transparent:
            m_transparent = !m_transparent;
            break;
	case TTKey_FlofRed:
	    if (m_decoder->findPage(page->floflink[0], -1, &page))
	    {
                m_curpage = page->pagenum;
		m_cursubpage = -1;
		m_curpage_showheader = true;
	    }
	    break;
	case TTKey_FlofGreen:
	    if (m_decoder->findPage(page->floflink[1], -1, &page))
	    {
		m_curpage = page->pagenum;
		m_cursubpage = -1;
		m_curpage_showheader = true;
	    }
	    break;
	case TTKey_FlofYellow:
	    if (m_decoder->findPage(page->floflink[2], -1, &page))
	    {
		m_curpage = page->pagenum;
		m_cursubpage = -1;
		m_curpage_showheader = true;
	    }
	    break;
	case TTKey_FlofBlue:
	    if (m_decoder->findPage(page->floflink[3], -1, &page))
	    {
		m_curpage = page->pagenum;
		m_cursubpage = -1;
		m_curpage_showheader = true;
	    }
	    break;
    }

    // XXX - Wrap?
    if (m_curpage < 0x100)
        m_curpage = 0x100;
    if (m_curpage > 0x899)
        m_curpage = 0x899;

    if (!numeric_input)
    {
        m_pageinput[0] = (m_curpage / 256) + '0';
        m_pageinput[1] = ((m_curpage % 256) / 16) + '0';
        m_pageinput[2] = (m_curpage % 16) + '0';
    }

    pageUpdated(m_curpage);
}

char TeletextView::charConversion(char ch, int lang)
{
    int c = 0;
    for (int j = 0; j < 14; j++)
    {
        c = ch & 0x7f;
        if ( c == lang_chars[0][j])
            ch = lang_chars[lang + 1][j];
    }
    return ch;
}

//
//Decoder section
//

#define HAMMING84(d) ( ((d & 0x80) >> 4) | ((d & 0x20) >> 3) | ((d & 0x08) >> 2) | ((d & 0x02) >> 1) )
#define BAD_CHAR	0xb8	// substitute for chars with bad parity
#define PIL(day, mon, hour, min) \
	(((day) << 15) + ((mon) << 11) + ((hour) << 6) + ((min) << 0))
#define CC_SIZE 70

uint32_t decodeHamm24(uint8_t d1, uint8_t d2, uint8_t d3)
{
    uint32_t ret = 0;
    ret |= (d1 & 0x04) << 2;
    ret |= (d1 & 0x70) << 3;
    ret |= (d2 & 0x7F) << 4;
    ret |= (d3 & 0x7F) << 5;
    return ret;
}

int hamm8(uint8_t *p, int *err)
{
    int a = hammtab[p[0]];
    *err += a;
    return a & 15;
}

int hamm84(uint8_t *p, int *err)
{
    int a = hamm84tab[p[0]];
    
    if (a == 255)
        *err = 1;
    
    return a;
}
int hamm16(uint8_t *p, int *err)
{
    int a = hammtab[p[0]];
    int b = hammtab[p[1]];
    *err += a;
    *err += b;
    return (a & 15) | (b & 15) * 16;
}

int hamm24(uint8_t *p, int *err)
{
    int e = hamm24par[0][p[0]] ^ hamm24par[1][p[1]] ^ hamm24par[2][p[2]];
    int x = hamm24val[p[0]] + p[1] % 128 * 16 + p[2] % 128 * 2048;

    *err += hamm24err[e];
    return x ^ hamm24cor[e];
}

int chk_parity(uint8_t *p, int n)
{
    int err;

    for (err = 0; n--; p++)
	if (hamm24par[0][*p] & 32)
	    *p &= 0x7f;
	else
	    *p = BAD_CHAR, err++;
    return err;
}

//ccdecode
char    *ratings[] = {"(NOT RATED)","TV-Y","TV-Y7","TV-G","TV-PG","TV-14","TV-MA","(NOT RATED)"};
int     rowdata[] = {11,-1,1,2,3,4,12,13,14,15,5,6,7,8,9,10};
char    *modes[]={"current","future","channel","miscellanious","public service","reserved","invalid","invalid","invalid","invalid"};
char	*specialchar[] = {"®","°","½","¿","(TM)","¢","£","o/~ ","à"," ","è","â","ê","î","ô","û"};
int     lastcode;
int     ccmode=1;               //cc1 or cc2
char    ccbuf[3][256];          //cc is 32 columns per row, this allows for extra characters
int     keywords=0;
char    *keyword[32];

VbiDecoder::VbiDecoder()
{
    m_seendata = false;
    m_waitpage = -1;

    for (int i=0; i<8; ++i) {
        QMutexLocker lock(&m_magazines[i].lock);

        m_magazines[i].pages.clear();
        m_magazines[i].current_page = NULL;
    }
    // fill Bitswap
    for (int i=0; i<256; i++)
    {
	bitswap[i]=0;
        for (int bit = 0; bit < 8; bit++)
	    if (i & (1 << bit))
		bitswap[i] |= (1 << (7-bit));
    }
}

VbiDecoder::~VbiDecoder()
{
}

void VbiDecoder::dumpPil(int pil)
{
    int day, mon, hour, min;

    day = pil >> 15;
    mon = (pil >> 11) & 0xF;
    hour = (pil >> 6) & 0x1F;
    min = pil & 0x3F;

    if (pil == PIL(0, 15, 31, 63))
        VERBOSE(VB_VBI," PDC: Timer-control (no PDC)");
        else if (pil == PIL(0, 15, 30, 63))
            VERBOSE(VB_VBI," PDC: Recording inhibit/terminate");
        else if (pil == PIL(0, 15, 29, 63))
            VERBOSE(VB_VBI," PDC: Interruption");
        else if (pil == PIL(0, 15, 28, 63))
            VERBOSE(VB_VBI," PDC: Continue");
        else if (pil == PIL(31, 15, 31, 63))
            VERBOSE(VB_VBI," PDC: No time");
        else
            VERBOSE(VB_VBI,QString(" PDC: %1, 200X-%2-%3 %4:%5")
            .arg(pil).arg(mon).arg(day).arg(hour).arg(min));
}

int VbiDecoder::decodeVps(uint8_t *buf)
{
    static char pr_label[20];
    static char label[20];
    static int l = 0;
    int cni, pcs, pty, pil;
    int c;
    //unsigned char *buf = p;

    VERBOSE(VB_VBI,"\nVPS:");
    c = vbi_bit_reverse[buf[1]];

    if ((int8_t) c < 0) {
        label[l] = 0;
        memcpy(pr_label, label, sizeof(pr_label));
        l = 0;
    }
    c &= 0x7F;
    label[l] = printable(c);
    l = (l + 1) % 16;
    VERBOSE(VB_VBI, QString(" 3-10: %1 %2 %3 %4 %5 %6 %7 %8 (\"%9\")")
        .arg(buf[0]).arg(buf[1]).arg(buf[2]).arg(buf[3]).arg(buf[4])
	.arg(buf[5]).arg(buf[6]).arg(buf[7]).arg(pr_label));
    pcs = buf[2] >> 6;
    cni = + ((buf[10] & 3) << 10)
          + ((buf[11] & 0xC0) << 2)
          + ((buf[8] & 0xC0) << 0)
          + (buf[11] & 0x3F);
    pil = ((buf[8] & 0x3F) << 14) + (buf[9] << 6) + (buf[10] >> 2);
    pty = buf[12];
    VERBOSE(VB_VBI, QString("CNI: %1 PCS: %2 PTY: %3 ").arg(cni).arg(pcs).arg(pty));
    dumpPil(pil);

    return c & 0xf0;
}

int VbiDecoder::decodeWss(uint8_t *buf)
{
    return 1;
    static const int wss_bits[8] = {
        0, 0, 0, 1, 0, 1, 1, 1 };
    unsigned char parity;
    int wss = 0;
    int i;

    for (i = 0; i < 16; i++) {
        int b1 = wss_bits[buf[i] & 7];
        int b2 = wss_bits[(buf[i] >> 3) & 7];

        if (b1 == b2) return -1;
            wss |= b2 << i;
    }
    parity = wss & 15;
    parity ^= parity >> 2;
    parity ^= parity >> 1;

    fprintf(stderr,"WSS: %s; %s mode; %s color coding;\n"
        "      %s helper; reserved b7=%d; %s\n"
        "      open subtitles: %s; %scopyright %s; copying %s\n",
        formats[wss & 7],
        (wss & 0x10) ? "film" : "camera",
        (wss & 0x20) ? "MA/CP" : "standard",
        (wss & 0x40) ? "modulated" : "no",
        !!(wss & 0x80),
        (wss & 0x0100) ? "have TTX subtitles; " : "",
        subtitles[(wss >> 9) & 3],
        (wss & 0x0800) ? "surround sound; " : "",
        (wss & 0x1000) ? "asserted" : "unknown",
        (wss & 0x2000) ? "restricted" : "not restricted");

    if (!(parity & 1)) return -1;

    return wss;
}

int VbiDecoder::oddParity(uint8_t c)
{
    c ^= (c >> 4);
    c ^= (c >> 2);
    c ^= (c >> 1);

    return c & 1;
}

void VbiDecoder::decodeXds(uint8_t *buf)
{
    char c;

    VERBOSE(VB_VBI,QString("XDS: %1 %2").arg(buf[0]).arg(buf[1]));
    c = oddParity(buf[0]) ? buf[0] & 0x7F : '?';
    c = printable(c);
    putchar(c);
    c = oddParity(buf[1]) ? buf[1] & 0x7F : '?';
    c = printable(c);
    putchar(c);
    putchar('\n');
}

int VbiDecoder::ccDecoder(int b1, int b2, int len)
{
    int x=0,y=0,row;

    if ((b1&0x10) && (b2>0x1F)) {
        //codes are always transmitted twice 
        //(apparently not, ignore the second occurance)
        ccmode=((b1>>3)&1)+1;
        len = strlen(ccbuf[ccmode]);

        if (b2 & 0x40) { //preamble address code (row & indent)
            row=rowdata[((b1<<1)&14)|((b2>>5)&1)];
            if (len!=0)
                ccbuf[ccmode][len++]='\n';
                if (b2&0x10) //row contains indent flag
                    for (x=0;x<(b2&0x0F)<<1;x++)
                        ccbuf[ccmode][len++]=' ';
        }
        else {
            switch (b1 & 0x07) {
                case 0x00:      //attribute
                    VERBOSE(VB_VBI, QString("<ATTRIBUTE %1 %2").arg(b1).arg(b2));
                    fflush(stdout);
                    break;
                case 0x01:      //midrow or char
                    switch (b2&0x70) {
                        case 0x20: //midrow attribute change
                            switch (b2&0x0e) {
                                case 0x00: //italics off
                                    strcat(ccbuf[ccmode],"\33[0m ");
                                    break;
                                case 0x0e: //italics on
                                    strcat(ccbuf[ccmode],"\33[36m ");
                                    break;
                            }
                            if (b2&0x01) //underline
                                strcat(ccbuf[ccmode],"\33[4m");
                            else
                                strcat(ccbuf[ccmode],"\33[24m");
                            break;
                        case 0x30: //special character..
                            strcat(ccbuf[ccmode],specialchar[b2&0x0f]);
                            break;
                    }
                    break;
                case 0x04:      //misc
                case 0x05:      //misc + F
                    //printf("ccmode %d cmd %02x\n",ccmode,b2);
                    switch (b2) {
                        case 0x21: //backspace
                            ccbuf[ccmode][len--]=0;
                            break;
                            /* these codes are insignifigant if we're ignoring positioning */
                        case 0x25: //2 row caption
                        case 0x26: //3 row caption
                        case 0x27: //4 row caption
                        case 0x29: //resume direct caption
                        case 0x2B: //resume text display
                        case 0x2C: //erase displayed memory
                            break;
                        case 0x2D: //carriage return
                            if (ccmode==2)
                                break;
                        case 0x2F: //end caption + swap memory
                        case 0x20: //resume caption (new caption)
                            if (!strlen(ccbuf[ccmode]))
                                break;
                            for (x=0;x<strlen(ccbuf[ccmode]);x++)
                                for (y=0;y<keywords;y++)
                                    if (!strncasecmp(keyword[y], ccbuf[ccmode]+x, strlen(keyword[y])))
                                        //printf("%s\n",ccbuf[ccmode]);
                                        /* FALL */
                                        case 0x2A: //text restart
                                        case 0x2E: //erase non-displayed memory
                                            memset(ccbuf[ccmode],0,255);
                                            break;
                    }
                    break;
                case 0x07:      //misc (TAB)
                    for(x=0;x<(b2&0x03);x++)
                        ccbuf[ccmode][len++]=' ';
                        break;
            }
        }
    }
    return 0;
}

int VbiDecoder::decodeCaption(uint8_t *buf, int line)
{
    static int xds_transport = 0;
    char c = buf[0] & 0x7F;
    static char cc[CC_SIZE + 1];
    static int cc_idx;
    VERBOSE(VB_VBI,"decodeCaption");
    
    if (line >=270) {/* in field 2 */
        if (oddParity(buf[0]) && (c >= 0x01 && c <= 0x0F)) {
            decodeXds(buf);
            xds_transport = (c != 0x0F);
        } else if (xds_transport) {
            decodeXds(buf);
        }
    }

    c = oddParity(buf[0]) ? buf[0] & 0x7F : '?';
    c = printable(c);
    if (cc_idx >= CC_SIZE) {
        cc_idx = CC_SIZE - 2;
        memcpy(cc, cc + 2, cc_idx);
    } 
    cc[cc_idx++] = c;
    c = oddParity(buf[1]) ? buf[1] & 0x7F : '?';
    c = printable(c);
    cc[cc_idx++] = c;
    cc[cc_idx] = 0;

    ccDecoder(buf[0], buf[1], cc_idx);

    if (cc_idx == CC_SIZE) {
        VERBOSE(VB_VBI,QString("CC: %1 %2 %3").arg(buf[0]).arg(buf[1]).arg(cc));
        memset(cc, 0, CC_SIZE);
        cc_idx = 0;

    } 

    return 0;
}


void VbiDecoder::reset(void)
{
    // XXX - lock?
    for (int i=0; i<8; ++i)
    {
        QMutexLocker lock(&m_magazines[i].lock);
        m_magazines[i].pages.clear();
        m_magazines[i].current_page = NULL;
        VERBOSE(VB_VBI, QString("resetting magazine %1, %2 pages left")
	    .arg(i).arg(m_magazines[i].pages.size()));
    }
}

// if subpage = -1, find the first subpage
bool VbiDecoder::findPage(int page, int subpage, TeletextPage **ttpage)
{
    int mag = page / 256;
    if (mag > 8 || mag < 1)
        return false;
    if (mag == 8)
        mag = 0;

    QMutexLocker lock(&m_magazines[mag].lock);

    if (subpage == -1)
    {
        std::map<int, TeletextPage>::iterator iter;
	int j=0; // TODO: get the newest subpage
	do
	{
            iter = m_magazines[mag].pages.find(j * 256 + page % 256);
	    j++;
	}
        while ((iter == m_magazines[mag].pages.end()) && (j < 10));    
	
	if (iter == m_magazines[mag].pages.end())
               return false;
	
	*ttpage = &iter->second;
    }
    else
    {
	std::map<int, TeletextPage>::iterator iter;
	iter = m_magazines[mag].pages.find(subpage * 256 + page % 256);
        if (iter == m_magazines[mag].pages.end()) 
            return false;
	*ttpage = &iter->second;
    }

    return true;
}

#define PG_ACTIVE	0x100	// currently fetching this page

int VbiDecoder::decodeTeletext(uint8_t *buf,int vbimode)
{
    m_seendata = true;
    int err = 0;
    int latin1 = -1;
    uint8_t magazine,packet,header;
    int zahl1;

    switch (vbimode)
    {
	case VBI_IVTV:
            header = hamm16(buf, &err);

            if (err & 0xf000)
               return -4;
    
            magazine = header & 7;
            packet = (header >> 3) & 0x1f;

	    buf += 2;
	    break;
	    
	case VBI_DVB:
	case VBI_DVB_ST:
	    zahl1 = hamm84(buf,&err) * 16 + hamm84(buf+1,&err);
        
	    magazine = 0;
	    if (buf[0] & 0x40)
	        magazine += 1;
	    if (buf[0] & 0x10)
	        magazine += 2;
	    if (buf[0] & 0x04)
                magazine += 4;

            packet = 0;
	    if (buf[0] & 0x01)
	        packet += 1;
	    if (buf[1] & 0x40)
	        packet += 2;
	    if (buf[1] & 0x10)
	        packet += 4;
	    if (buf[1] & 0x04)
	        packet += 8;
	    if (buf[1] & 0x01)
	        packet += 16;

	    if (err == 1)
	    {
	        VERBOSE(VB_VBI,"VbiDecoder:Error in magazine or packet number!");
		return -4;
	    }
	    buf += 2;
	    break;

	default:
	    return -5; // error in vbimode
    }

    QMutexLocker lock(&m_magazines[magazine].lock);
    switch (packet) {
        case 0:
        { // Page Header
            TeletextPage *old_page = m_magazines[magazine].current_page;
            TeletextPage *new_page = NULL;

            int b1, b2, b3, b4;
            switch (vbimode)
	    {
	        case VBI_IVTV:
                    b1 = hamm16(buf, &err);	// page number
                    b2 = hamm16(buf+2, &err);	// subpage number + flags
                    b3 = hamm16(buf+4, &err);	// subpage number + flags
                    b4 = hamm16(buf+6, &err);	// language code + more flags
		    if (err & 0xf000)
			return -4;
	            break;

		case VBI_DVB:
		case VBI_DVB_ST:
		    b1 = hamm84(buf+1,&err)*16+hamm84(buf,&err);
	            b2 = hamm84(buf+3,&err)*16+hamm84(buf+2,&err);
	            b3 = hamm84(buf+5,&err)*16+hamm84(buf+4,&err);
	            b4 = hamm84(buf+7,&err)*16+hamm84(buf+6,&err);
	            if (err == 1)
	            {
                        VERBOSE(VB_VBI,"vbidecoder: Error in page number!");
		        m_magazines[magazine].current_page = NULL;
		        return -4;
	            }
		    break;

		default:
		    return -5; // error in vbimode
	    }
	    
	    VERBOSE(VB_VBI,QString("Page Header found: Magazine %1, Page Number %2").arg(magazine).arg(b1));
	  
	    m_serialmode = (b2 + b3 * 256) & 0x3f7f;
            if (m_serialmode)
                for (int i=0; i<8; ++i)
                    if (m_magazines[i].current_page)
                        old_page = m_magazines[i].current_page;


            if (old_page)
            {
                m_magazines[magazine].lock.unlock();
		pageUpdated(old_page->pagenum);
                m_magazines[magazine].lock.lock();
            }
            int subpgnum = (b2 + b3 * 256) & 0x3f7f;
            new_page = &m_magazines[magazine].pages[subpgnum * 256 + b1];
            new_page->pagenum =(magazine?: 8) * 256 + b1;
	    new_page->subpagenum = subpgnum;
            new_page->lang = "\0\4\2\6\1\5\3\7"[b4 >> 5] + (latin1 ? 0 : 8);
            new_page->flags = b4 & 0x1f;
            new_page->flags |= b3 & 0xc0;
            new_page->flags |= (b2 & 0x80) >> 2;
            new_page->lines = 1;
            new_page->flof = 0;

	    for (int i=0; i<6; i++)
		new_page->floflink[i] = 0;
	    
	    new_page->subtitle = false;
	    
	    if (vbimode == VBI_DVB_ST)
                new_page->subtitle = true; 
	    
	    for (int j = 0; j<8; j++)
		buf[j] = 32;
	    
	    if (vbimode == VBI_DVB || vbimode == VBI_DVB_ST)
	    {
	        for (int j = 8; j <40; j++)
	            buf[j]=bitswap[buf[j]];
	    }
	   
	    // now in TeletextView::drawPage
	    // charConversion(buf + 8, 32, new_page->lang);

            new_page->flags |= PG_ACTIVE;
            memcpy(new_page->data[0]+0, buf, 40);
            memset(new_page->data[0]+40, ' ', sizeof(new_page->data)-40);

	    if ((new_page->flags & 0xc1) || (vbimode==VBI_DVB_ST))
	        for (int j = 1; j < 24; j++) 
                    new_page->data[j][0]=255; 
	    
	    m_magazines[magazine].current_page = new_page;
	    headerUpdated(new_page->data[0],new_page->lang);
        }
        break;

    
        case 1 ... 24:
        { // Page Data
            if (!m_magazines[magazine].current_page) {
                return -1;
            }


            TeletextPage *page = m_magazines[magazine].current_page;
	    //page->lines |= 1 << packet;
/*
            TeletextPage *osdpage;
	    bool flof_update = FALSE;
	    if (!m_decoder->findPage(m_durpage, &osdpage))
	    {
	        for (int i=0; i<6; i++)
	            if (osdpage->floflink[i] = page->pagenum)
		        flof_update = TRUE;
	    }
	    if (flof_update)
	        pageUpdated(page->pagenum);
*/
            if (vbimode == VBI_DVB || vbimode == VBI_DVB_ST)
	    {
	        for (int j = 0; j < 40; j++)
                    buf[j]=bitswap[buf[j]];
	    }

	    // now in TeletextView::drawPage
	    // charConversion(buf, 40, page->lang);
            memcpy(page->data[packet], buf, 40);
	    //pageUpdated(page->pagenum);
        }        
        break;

        case 26:
        {   /* XXX TODO: Level 1.5, 2.5, 3.5
             *      Character location & override.
             * Level 2.5, 3.5
             *      Modifying display attributes.
             * All levels
             *      VCR Programming
             * See 12.3
             */
	     /*
	     if (!m_magazines[magazine].current_page)
		 return -1;
	     
	     TeletextPage *page = m_magazines[magazine].current_page;

             for (int i = 0; i < 43; i++)
	         fprintf(stderr,"%x ",buf[i]);
	     fprintf(stderr,"\n");
	     */
	}
        break;

        case 27:
        {   
	    // FLOF data (FastText)
	    if (!m_magazines[magazine].current_page)
		return -1;

	    int b1=0,b2=0,b3=0,x;
	    TeletextPage *page = m_magazines[magazine].current_page;

	    switch (vbimode)
	    {
		case VBI_IVTV:
	            b1 = hamm8(buf, &err);
	            b2 = hamm8(buf + 37, &err);
	            if (err & 0xf000)
		        return 4;
		    break;
		case VBI_DVB:
		case VBI_DVB_ST:
		    b1 = hamm84(buf, &err);
		    b2 = hamm84(buf + 37, &err);
		    if (err == 1)
			return 4;
		    break;
	    }
	    
	    if (b1 != 0 || not(b2 & 8))
		return 0;

	    for (int i = 0; i < 6; ++i) {
		err = 0;
		switch (vbimode)
		{
		    case VBI_IVTV:
		        b1 = hamm16(buf+1+6*i, &err);
		        b2 = hamm16(buf+3+6*i, &err);
		        b3 = hamm16(buf+5+6*i, &err);
		        if (err & 0xf000)
		            return 1;
		        
			break;
		    case VBI_DVB:
		    case VBI_DVB_ST:
			b1 = hamm84(buf+2+6*i, &err) * 16 + hamm84(buf+1+6*i, &err);
			b2 = hamm84(buf+4+6*i, &err) * 16 + hamm84(buf+3+6*i, &err);
			b3 = hamm84(buf+6+6*i, &err) * 16 + hamm84(buf+5+6*i, &err);
			if (err == 1)
			    return 1;
		
			break;
		}
		x = (b2 >> 7) | ((b3 >> 5) & 0x06);
		page->floflink[i] = ((magazine ^ x) ?: 8) * 256 + b1;

            }
	    page->flof = 1;
        }
        break;

	case 31:
	{   // XXX TODO private streams
	    if (!m_magazines[magazine].current_page)
		return -1;
	    
//          TeletextPage *page = m_magazines[magazine].current_page;
//	    for (int i=0;i<43;i++)
//		fprintf(stderr,"%02x ",buf[i]);
//	    fprintf(stderr,"\n");
	}
	break;
	
        default:
        {
            // XXX TODO other packet codes...
	    if (!m_magazines[magazine].current_page)
	        return -1;

//	    TeletextPage *page = m_magazines[magazine].current_page;
 //           for (int i=0;i<43;i++)
//		fprintf(stderr,"%02x ",buf[i]);
//	    fprintf(stderr,"\n");
	    //fprintf(stderr,"VbiDecoder: packet ID = %d \n", packet);
        }
   }
   return 0;
}

void VbiDecoder::processIvtvData(uint8_t *buf)
{
        long linemask[2] = { 0, 0 };
//      int wss;
	int i, l, id2;
        int field_lines=625; //what is this?
        uint8_t data[43];

        // XXX TODO: nomally we should check on 'itv0' or 'ITV0' 
        // but as we already check the stream type with the first
        // character we will do the check on the remaining chars here...

        if (!memcmp(buf, "tv0", 3)) {
            memcpy(linemask, buf + 3, 8);
            buf += 11;
        }
        else if (!memcmp(buf, "TV0", 3)) {
            linemask[0] = 0xffffffff;
            linemask[1] = 0xf;
            buf += 3;
        }
        else {
            VERBOSE(VB_VBI,"VbiDecoder: unknown VBI data stream");
            return;
        }

        for (i = 0; i < 36; i++) {
            int err = 0;
  
            if (i < 32 && !(linemask[0] & (1 << i))) continue;
            if (i >= 32 && !(linemask[1] & (1 << (i - 32)))) continue;
            l = (i < 18) ? i + 6 : i - 18 + 6 + field_lines;
            id2 = *buf & 0xf;
            switch (id2) {
                case VBI_TYPE_TELETEXT:
                    memcpy(data, buf+1, 42);
                    err = decodeTeletext(data,VBI_IVTV);
                    break;
                case VBI_TYPE_CC:
                    err = decodeCaption(buf+1,l);
                    break;
                case VBI_TYPE_VPS:
                    err = decodeVps(buf+1);
                    break;
                case VBI_TYPE_WSS:
                    err = decodeWss(buf+1);
                    break;
            }
            buf += 43;
        }
        return;
}

void VbiDecoder::processDVBData(uint8_t *buf, int len)
{
    uint8_t data[43];
    int j = 0;
    do
    {
	if (*buf==16)  // Why???
	{
	    buf++;
	    j++;
	}
	if (*buf == 255) // Why???
        {
	    buf += 3;
	    j += 3;
	}

	int k = *buf;  // data_unit_id: 3 - subtitle

        if ((k==2) || (k==3))
        {
            // DVB Teletext
	    buf += 3;
	    j += 3;
	    int err=0;

	    memcpy(data,buf+1,42);
	    
	    if (k==2)
	        err = decodeTeletext(data,VBI_DVB); 
	    else
		err = decodeTeletext(data,VBI_DVB_ST); 
        
	}
	else
	{
	    VERBOSE(VB_VBI,QString("Unknown descriptor: %1").arg(k));
	}
	buf += 43;
	j += 43;
    }
    while (j < len);
	    
    return;
}
