#ifndef VBIDECODER_H_
#define VBIDECODER_H_

#include <stdint.h>
#include <map>
#include <qwaitcondition.h>
#include <qobject.h>
#include <qmutex.h>

class VbiDecoder;
class TeletextPage;

#define MAX_PAGE_WIDTH 56
#define TTCOLOR_BLACK   0
#define TTCOLOR_RED     1
#define TTCOLOR_GREEN   2
#define TTCOLOR_YELLOW  3
#define TTCOLOR_BLUE    4
#define TTCOLOR_MAGNETA 5
#define TTCOLOR_CYAN    6
#define TTCOLOR_WHITE   7
#define TTCOLOR_TRANSPARENT 8

enum TTKey {
    TTKey_0 = 0, TTKey_1, TTKey_2, TTKey_3, TTKey_4,
    TTKey_5, TTKey_6, TTKey_7, TTKey_8, TTKey_9,
    TTKey_NextPage, TTKey_PrevPage,
    TTKey_NextSubPage, TTKey_PrevSubPage,
    TTKey_Hold, TTKey_Transparent,
    TTKey_FlofRed, TTKey_FlofGreen, TTKey_FlofYellow, TTKey_FlofBlue
};

class TeletextPage
{
  public:
    int pagenum,subpagenum; // the wanted page number
    uint8_t charset;

    int lang;               // language code
    int flags;              // misc flags (see PG_xxx below)
    int errors;             // number of single bit errors in page
    long lines;             // 1 bit for each line received
    uint8_t data[25][40];   // page contents
    int flof;               // page has FastText links
    int floflink[6];        // FastText links (FLOF)
    bool subtitle;          // page is subtitle page

    std::map<int, TeletextPage> subpages;
};

class TeletextMagazine
{
  public:
    QMutex lock;
    TeletextPage *current_page;
    std::map<int, TeletextPage> pages;
};

class TeletextView
    : virtual public QObject
{
    Q_OBJECT

  public:
    TeletextView();
    ~TeletextView();
    
    void setDecoder(class VbiDecoder *);
    void setSideBar(int div);

    virtual void setBackgroundColor(int color) = 0;
    virtual void setForegroundColor(int color) = 0;

    virtual void drawBackground(int x, int y, bool transparent = false) = 0;
    virtual void drawCharacter(int x, int y, QChar ch, int doubleheight) = 0;
    virtual void drawMosaic(int x, int y, int code, int doubleheight) = 0;

    virtual void newsFlash() = 0;

    int getCurrentPage(void) { return m_curpage; };

    bool isVisible(void) { return m_visible; };

  public slots:
    virtual void keyPress(enum TTKey key);
    void decoderDestroyed() { /*LOCK*/ m_decoder = NULL; };

  protected slots:
    virtual void pageUpdated(int page) = 0;
    virtual void headerUpdated(uint8_t* page, int lang) = 0;

  protected:
    void drawLine(uint8_t* page, int row, int lang);
    void drawHeader(uint8_t* page, int lang);
    void drawPage(void);
    char charConversion(char ch, int lang);
    
  protected:
    class VbiDecoder *m_decoder;

    char m_pageinput[3];
    int m_curpage;
    int m_cursubpage;
    bool m_curpage_showheader;
    
    bool m_pageupdated;
    uint8_t* m_curheader;

    int m_sidebar;
    
    bool m_visible;
    bool m_transparent;
    bool m_header;
    
    int m_tt_cols;
    int m_tt_rows;

};

class VbiDecoder
    : public QObject
{
    Q_OBJECT

    friend class TeletextPage;

  public:
    VbiDecoder();
    ~VbiDecoder();

    void processIvtvData(uint8_t *buf);
    void processDVBData(uint8_t *buf, int len);

    void dumpPil(int pil);
    int decodeTeletext(uint8_t *buf, int vbimode);
    int decodeWss(uint8_t *buf);
    int decodeVps(uint8_t *buf);
    int decodeCaption(uint8_t *buf, int line);
    int oddParity(uint8_t c);
    void decodeXds(uint8_t *buf);
    int ccDecoder(int b1, int b2, int len);
    
    int getStartPage(void) { return m_startpage==0?0x100:m_startpage; };

    // Utility functions for viewers
    
  public slots:
    void reset(void);

    bool findPage(int page, int subpage, TeletextPage **ttpage);

  signals:
    void pageUpdated(int page);
    void headerUpdated(uint8_t* page, int lang);

    void wss(int aspect);
    void cc(void);

  protected:
    void releasePage(class TeletextPage *);

    int ebuData(uint8_t *buf, int len);

  protected:
    bool m_seendata;
    bool m_serialmode;
    int m_startpage;

    int m_waitpage;
    QWaitCondition m_waitcondition;

    TeletextMagazine m_magazines[8];
  
  private:
    unsigned char bitswap[256];
};

#endif
