/*
 * Samo's SA4250HD firewire channel changer by majoridiot
 * with node selection added by Bradley S. Corsello
 *
 * requires: libavc1394-dev libraw1394-dev 
 *
 * compile with: gcc -o sa4250_ch sa4250_ch.c -lrom1394 -lavc1394 -lraw1394
 *
 * based on mythtv source code and
 *
 * sa3250ch - an external channel changer for SA3250HD Tuner 
 * Based off 6200ch.c by Stacey D. Son
 * 
 * Copyright 2004,2005 by Stacey D. Son <mythdev@son.org> 
 * Copyright 2005 Matt Porter <mporter@kernel.crashing.org>
 * Portions Copyright 2006 Chris Ingrassia <chris@spicecoffee.org> (SA4200 and Single-digit command mode)
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <libavc1394/rom1394.h>
#include <libavc1394/avc1394.h>
#include <libraw1394/raw1394.h>
#include <sys/types.h>
#include <stdio.h>
#include <errno.h>
#include <stdlib.h>
#include <string.h>

/* SA42xxHD IDs */
#define SA4200HD_VENDOR_ID1     0x000014f8
#define SA4200HD_MODEL_ID1      0x00001072
#define SA4250HD_VENDOR_ID1	0x00001cea  /* samo's stb */
#define SA4250HD_MODEL_ID1	0x000010cc  /* samo's stb */

/* add additional vendor and model id's here- addition needed to if statement starting @ line 134 below */

#define AVC1394_SA3250_COMMAND_CHANNEL 0x000007c00   
#define AVC1394_SA3250_OPERAND_KEY_PRESS 0xe7
#define AVC1394_SA3250_OPERAND_KEY_RELEASE 0x67 

#define CTL_CMD0 AVC1394_CTYPE_CONTROL | AVC1394_SUBUNIT_TYPE_PANEL | \
        AVC1394_SUBUNIT_ID_0 | AVC1394_SA3250_COMMAND_CHANNEL
#define CTL_CMD1 (0x04 << 24)
#define CTL_CMD2 0xff000000

#define STARTING_NODE 0

void usage()
{
   fprintf(stderr, "Usage: sa4250_ch_new [-v] [-n <node>] <channel_num>\n");
   fprintf(stderr, "  -v : Verbose Mode\n");
   fprintf(stderr, "  -n : Node to change - default 0\n");
   exit(1);
}

int main (int argc, char *argv[])
{
   rom1394_directory dir;
   int device = -1;
   int single = 0;
   int i;
   int verbose = 0;
   int dig[3];
   int chn = 708;
   int unode = 0;

   if (argc < 2) 
      usage();

  for(i = 1; i < argc; ++i) {
      if ((argv[i][0] == '-') && (strlen(argv[i]) > 1)) {
        switch(argv[i][1]) {
            case 'v':
                verbose = 1;
                break;
            case 's':
                single = 1;
                break;
            case 'n':
                unode = atoi(argv[++i]);
		break;
            default:
                fprintf(stderr, "WARNING: Unknown option \'%c\', ignoring", argv[i][1]);
        }
      }
      else {
          chn = atoi(argv[i]);
      }
  }

#ifdef RAW1394_V_0_8
   raw1394handle_t handle = raw1394_get_handle();
#else
   raw1394handle_t handle = raw1394_new_handle();
#endif

   if (!handle) {
      if (!errno) {
         fprintf(stderr, "Not Compatible!\n");
      } else {
         perror("Couldn't get 1394 handle");
         fprintf(stderr, "Is ieee1394, driver, and raw1394 loaded?  Are /dev/raw1394 permissions set correctly?\n");
      }
      exit(1);
   } 

   if (raw1394_set_port(handle, 0) < 0) {
      perror("ERROR-- could not set port");
      raw1394_destroy_handle(handle);
      exit(1);
   }

   int nc = raw1394_get_nodecount(handle);
      if (rom1394_get_directory(handle, unode, &dir) < 0) {
         fprintf(stderr,"ERROR reading config rom directory for node %d\n", unode);
         raw1394_destroy_handle(handle);
    	 exit(1);
      }

      if (verbose) 
         printf("node %d: vendor_id = 0x%08x model_id = 0x%08x\n", 
                 unode, dir.vendor_id, dir.model_id); 

/* add new vendor and model ids	to if stanza below */
	
      if ( ((dir.vendor_id == SA4250HD_VENDOR_ID1) &&
            (dir.model_id == SA4250HD_MODEL_ID1))  || 
	  ((dir.vendor_id == SA4200HD_VENDOR_ID1) &&
	    (dir.model_id == SA4200HD_MODEL_ID1))) { 
            device = unode;
      }
    
   if (device == -1) {
        fprintf(stderr, "Could not find SA42XXHD on the 1394 bus!\n");
	fprintf(stderr, "Try running again with -v and check source code for matching Vendor ID and Model ID-\n");
	fprintf(stderr, "Add if necessary and recompile.\n"); 
        raw1394_destroy_handle(handle);
        exit(1);
   }

   if (verbose)
        printf("Device acquired on node %d\n", device);
        printf("Changing channel %d\n", chn);
        
	quadlet_t cmd[3] =
        {
            CTL_CMD0 | AVC1394_SA3250_OPERAND_KEY_PRESS,
            CTL_CMD1 | (chn << 8),
            CTL_CMD2,
        };        

       if (verbose)
            printf("AV/C Command: cmd0=0x%08x cmd1=0x%08x cmd2=0x%08x\n",
                   cmd[0], cmd[1], cmd[2]);
       avc1394_transaction_block(handle, device, cmd, 3, 1);       

   raw1394_destroy_handle(handle);

   exit(0);
}
