/*
        MythWeather
        Version 0.8
        January 8th, 2003

        By John Danner & Dustin Doris

        Note: Portions of this code taken from MythMusic

*/

#include <qlabel.h>
#include <fstream>
#include <iostream>
#include <qdatetime.h>
#include <qlistview.h>
#include <qtimer.h>
#include <qimage.h>
#include <qregexp.h>
#include <qnetwork.h>
#include <qstringlist.h>
#include <qheader.h>
#include <unistd.h>
#include <qurl.h>
#include <qdir.h>
#include <qfile.h>

#include "weatherdata.h"
#include "weather.h"

#include <mythtv/mythcontext.h>
#include <mythtv/httpcomms.h>

using namespace std;

WeatherData::WeatherData(class Weather *weather, QString locale, int pwantAnimated)
{
    parent = weather;

    httpData = "";
    weatherTimeoutInt = gContext->GetNumSetting("WeatherTimeout", 20);
    weatherTimeoutInt *= 1000;
    stopProcessing = false;

    wantAnimated = pwantAnimated;

    debug = false;
    if (parent && parent->debug) debug = parent->debug;

    convertData = false;
    if (parent && parent->convertData) convertData = true;

    this->locale = locale;

    urlTimer = new QTimer(this);
    connect(urlTimer, SIGNAL(timeout()), SLOT(weatherTimeout()));
}

WeatherData::~WeatherData()
{
    delete urlTimer;
}

void WeatherData::setLocation(QString newLocale)
{
    locale = newLocale;
}

QString WeatherData::GetFileName(QString type) {
    QString fileprefix = MythContext::GetConfDir();

    QDir dir(fileprefix);
    if (!dir.exists())
        dir.mkdir(fileprefix);

    fileprefix += "/MythWeather";
    QString sFile = "";
    if (type == "basedata") {
        sFile = fileprefix + "/basedata.html";
    } else if (type == "metadata") {
        sFile = fileprefix + "/metadata";
    } else if (type == "staticradar") {
        sFile = fileprefix + "/radar.jpg";
    } else if (type == "animradarpattern") {
        sFile = fileprefix + "/radar%1.jpg";
    } else {
	fprintf (stderr, "Unexpected query for weatherdata filename for %s\n",
				type.ascii());
    }
    return sFile;
}

bool WeatherData::LoadData()
{
    QString sFile = GetFileName("metadata");
    QFile metadata(sFile);
    if (!metadata.exists()) return false;
    if (!metadata.open(IO_ReadOnly)) return false;

    QString dataLocale, dataUpdate;
    if (metadata.readLine(dataLocale, 1024) <= 0) return false;
    int dLen = dataLocale.length();
    if (dLen < 1) return false;
    if (dataLocale[dLen-1] == '\n') dataLocale.truncate(dLen-1);
    if (dataLocale != locale) return false;
    if (metadata.readLine(dataUpdate, 1024) <= 0) return false;
    uint updatedTime = dataUpdate.toUInt();

    QDateTime upTime;
    upTime.setTime_t(updatedTime);

    int deltaTime = upTime.secsTo(QDateTime::currentDateTime());
    // Only want data that is less than 1.5 hours old
    if (deltaTime > 3600*1.5) return false;

    updated = upTime.toString(gContext->GetSetting("dateformat") +
               " " + gContext->GetSetting("timeformat"));
    int upLen = updated.length();
    if (upLen < 1) return false;
    if (updated[upLen-1] == '\n') updated.truncate(upLen-1);

    sFile = GetFileName("basedata");
    QFile data(sFile);
    if (!data.exists()) return false;
    if (!data.open(IO_ReadOnly)) return false;

    httpData = "";
    for (QString line; data.readLine(line, 1024) > 0; ) {
        httpData += line;
    }
    InterpretData();

    return true;
}

bool WeatherData::UpdateBaseData()
{
    bool haveData = false;

    stopProcessing = false;
    urlTimer->start(weatherTimeoutInt);
    haveData = GetBaseData();

    urlTimer->stop();

    InterpretData();

    return haveData;
}

QString WeatherData::GetString(QString tag)
{
    QString data;
    int start = 0;
    int len = 0;
    
    start = httpData.find(tag, 0);
    len = httpData.find("\"", start + tag.length() + 4) - start - tag.length() - 4;
    
    data = httpData.mid(start + tag.length() + 4, len);

    return data;
}

int WeatherData::GetInt(QString tag)
{
    QString data;
    int start = 0;
    int len = 0;
    
    start = httpData.find(tag, 0);
    len = httpData.find("\"", start + tag.length() + 4) - start - tag.length() - 4;
    data = httpData.mid(start + tag.length() + 4, len);
    int ret = data.toInt();

    return ret;
}

float WeatherData::GetFloat(QString tag)
{
    QString data;
    int start = 0;
    int len = 0;

    start = httpData.find(tag, 0);
    len = httpData.find("\"", start + tag.length() + 4) - start - tag.length() - 4;
    data = httpData.mid(start + tag.length() + 4, len);
    float ret = data.toFloat();

    return ret;
}

bool WeatherData::GetBaseData()
{
    QString weatherDataURL = "http://www.msnbc.com/m/chnk/d/weather_d_src.asp?acid=" + locale;

    VERBOSE(VB_NETWORK, QString("Grabbing weather from: %1").arg(weatherDataURL));

    httpData = HttpComms::getHttp(weatherDataURL, weatherTimeoutInt, 3, 3);

    if (httpData.find("this.swAcid = \"\";") != -1 ||
        httpData.find("<html>") != -1 ||
        httpData.find("Microsoft VBScript runtime") != -1 ||
        httpData.find("Internal Server Error") != -1  ||
        httpData.find("Bad Request", 0) != -1)
    {
        VERBOSE(VB_IMPORTANT, "MythWeather: Invalid area ID or server error.");
        if (debug)
            cerr << "MythWeather: HTTP Data Dump: " + httpData << endl;
        
        return false;
    }

    //updated = GetString("this.swLastUp");
    updated = QDateTime::currentDateTime().toString(gContext->GetSetting("dateformat") +
               " " + gContext->GetSetting("timeformat"));

    QString dFile = GetFileName("metadata");
    QFile metadata(dFile);
    if (metadata.open (IO_WriteOnly)) {
	QTextStream stream(&metadata);
	stream << locale << endl;
	stream << QDateTime::currentDateTime().toTime_t() << endl;
        metadata.close();
    }

    dFile = GetFileName("basedata");
    QFile basedata(dFile);
    if (basedata.open (IO_WriteOnly)) {
	QTextStream stream(&basedata);
	stream << httpData << endl;
        basedata.close();
    }

     return true;
}

bool WeatherData::GetStaticRadarMap()
{
    QString weatherMapLink1URL = QString("http://www.weather.com/weather/map/%1"
                                    "?from=LAPmaps&setcookie=1 HTTP/1.1\n"
                                    "Connection: close\nHost: www.weather.com\n\n\n")
                                    .arg(locale);

    QString tempData = HttpComms::getHttp(weatherMapLink1URL, weatherTimeoutInt, 3, 3);

    QString mapLoc = parseData(tempData, "if (isMinNS4) var mapNURL = \"", "\";");
    if (mapLoc == "<NULL>")
        return true;
     
    QString weatherMapLink2URL = "http://www.weather.com/" + mapLoc;
     
    if (debug)
        cerr << "MythWeather: Grabbing Weather Map Link (part 2) From: " 
             << weatherMapLink2URL << endl;
    
    VERBOSE(VB_NETWORK, QString("Grabbing weather map(2) from: %1").arg(weatherMapLink2URL));
              
    QString tempData2 = HttpComms::getHttp(weatherMapLink2URL, weatherTimeoutInt, 3, 3);

    QString imageLoc = parseData(tempData2, "<IMG NAME=\"mapImg\" SRC=\"http://image.weather.com", "\"");
    if (imageLoc == "<NULL>")
    {
        VERBOSE(VB_IMPORTANT, "MythWeather: Warning: Failed to find link to map image.");
        return false;
    }

    QString fileprefix = MythContext::GetConfDir();

    QDir dir(fileprefix);
    if (!dir.exists())
        dir.mkdir(fileprefix);

    fileprefix += "/MythWeather";

    dir = QDir(fileprefix);
    if (!dir.exists())
        dir.mkdir(fileprefix);

    if (debug)
        cerr << "MythWeather: Map File Prefix: " << fileprefix << endl;

    if (debug)
        cerr << "MythWeather: Copying Map File from Server (" << imageLoc << ")...";
    
    VERBOSE(VB_NETWORK, QString("MythWeather: Copying map file from server (%1)").arg(imageLoc));
     
    QString sFilename = GetFileName("staticradar");
    QString sURL("http://image.weather.com/" + imageLoc);

    if (!HttpComms::getHttpFile(sFilename, sURL, weatherTimeoutInt))
        cerr << "Failed to download image from:" << sURL << endl;

    if (debug)
        cerr << "Done.\n";

    return true;
}

bool WeatherData::GetAnimatedRadarMap()
{
     // find radar maps url's
     QString sURL = "http://www.weather.com/weather/map/" + locale +
        "?name=index_large_animated&day=1";
     QString tempData = "";

     if (debug)
        cerr << "MythWeather: Grabbing Weather Map Link (part 1) From: "
              << sURL << endl;
     tempData = HttpComms::getHttp(sURL, weatherTimeoutInt, 3, 3);

     QString mapLoc = parseData(tempData, "if (isMinNS4) var mapNURL = \"", "\";");
     if (mapLoc == "<NULL>")
         return false;

     mapLoc = "http://www.weather.com/" + mapLoc;
     if (debug)
        cerr << "MythWeather: Grabbing Weather Map Link (part 2) From: "
           << mapLoc << endl;

     tempData = HttpComms::getHttp(mapLoc, weatherTimeoutInt, 3, 3);

     QString imageLoc = parseData(tempData, "var thisMap = ['", "']");
     if (imageLoc == "<NULL>")
     {
         if (debug)
             cerr << "MythWeather: Warning: Failed to find link to map image.\n";

         return false;
     }

     QString fileprefix = MythContext::GetConfDir();

     QDir dir(fileprefix);
     if (!dir.exists())
         dir.mkdir(fileprefix);

     fileprefix += "/MythWeather";

     dir = QDir(fileprefix);
     if (!dir.exists())
         dir.mkdir(fileprefix);

     if (debug)
         cerr << "MythWeather: Map File Prefix: " << fileprefix << endl;

     // delete existing radar maps
     QString radarpattern = GetFileName("animradarpattern");
     for (int x = 1; x <= 6; x++)
         QFile::remove(radarpattern.arg(x));

     if (debug)
         cerr << "MythWeather: Copying Map Files from Server (" << imageLoc << ")...\n";

     for (int x = 1; x <= 6; x++)
     {
         QString sFile = radarpattern.arg(x);
         sURL = QString("http://image.weather.com" + imageLoc + "%1L.jpg").arg(x);
         if (!HttpComms::getHttpFile(sFile, sURL, weatherTimeoutInt))
             cerr << "Failed to download image from:" << sURL << endl;
     }

     if (debug)
         cerr << "MythWeather: Download radar images done.\n";

     return true;
}

QString WeatherData::parseData(QString data, QString beg, QString end)
{
    QString ret;

    if (debug == true)
    {
        cout << "MythWeather: Parse HTML : Looking for: " << beg << ", ending with: " << end << endl;
        if (data.length() == 0)
        {
            VERBOSE(VB_IMPORTANT, "MythWeather: Parse HTML: No data!");
            ret = "<NULL>";
            return ret;
        }
    }
    
    int start = data.find(beg) + beg.length();
    int endint = data.find(end, start + 1);
    if (start != -1 && endint != -1)
    {
        ret = data.mid(start, endint - start);
        if (debug == true)
            cout << "MythWeather: Parse HTML : Returning : " << ret << endl;
        return ret;
    }
    else
    {
        if (debug == true)
            VERBOSE(VB_IMPORTANT, "MythWeather: Parse HTML: Parse Failed...returning <NULL>");
        ret = "<NULL>";
        return ret;
    }
}

void WeatherData::InterpretData()
{
    city = GetString("this.swCity");
    state = GetString("this.swSubDiv");
    country = GetString("this.swCountry");
    curTemp = GetString("this.swTemp");
    if (curTemp.length() == 0)
    {
        curTemp = "-na-";
        updated = updated + tr(" (Not All Information Available)");
    }
    else
    {
        if (convertData == true)
        {
            char tempHold[32];
            double tTemp = curTemp.toDouble();
            double nTemp = (double)(5.0/9.0)*(tTemp - 32.0);
            sprintf (tempHold, "%d", (int)nTemp);
            curTemp = tempHold;
        }
    }

    curIcon = GetString("this.swCIcon");
    curWind = GetString("this.swWindS");
    if (convertData == true)
    {
        char tempHold[32];
        double tTemp = curWind.toDouble();
        double nTemp = (double)(tTemp*1.6);
        sprintf (tempHold, "%d", (int)nTemp);
        curWind = tempHold;
    }

    winddir = GetString("this.swWindD");

    barometer = GetString("this.swBaro");
    if (convertData == true)
    {
        char tempHold[32];
        double tTemp = barometer.toDouble();
        double nTemp = (double)(tTemp*3.386);
        sprintf (tempHold, "%.1f", nTemp);
        barometer = tempHold;
    }
    
    curHumid = GetString("this.swHumid");
    curFeel = GetString("this.swReal");
    if (convertData == true)
    {
        char tempHold[32];
        double tTemp = curFeel.toDouble();
        double nTemp = (5.0/9.0)*(tTemp - 32.0);
        sprintf (tempHold, "%d", (int)nTemp);
        curFeel = tempHold;
    }
    
    uvIndex = GetString("this.swUV");
    visibility = GetString("this.swVis");
    if (convertData == true && visibility.toDouble() != 999.0)
    {
        char tempHold[32];
        double tTemp = visibility.toDouble();
        double nTemp = (double)(tTemp*1.6);
        sprintf (tempHold, "%.1f", nTemp);
        visibility = tempHold;
    }
    
    description = GetString("this.swConText");
    if (description.length() == 0)
        description = curIcon;
    
    QString forecastData;
    forecastData = GetString("this.swFore");

    QStringList holdings = QStringList::split("|", forecastData);

    int years, mons, days;

    int dayNum = (holdings[0]).toInt();
    QDate curDay(QDate::currentDate());
    int curDayNum = curDay.dayOfWeek();
    if (curDayNum == 7)
        curDayNum = 1;
    else
        curDayNum++;

    if (curDayNum != dayNum)
        pastTime = true;
    else
        pastTime = false;

    for (int i = 5; i < 9; i++)
    {
        QStringList dates = QStringList::split("/", holdings[i]);
        years = dates[2].toInt();
        mons = dates[0].toInt();
        days = dates[1].toInt();
        QDate doDate(years, mons, days);
        switch (doDate.dayOfWeek())
        {
            case 1: date[i - 5] = QString("MON"); break;
            case 2: date[i - 5] = QString("TUE"); break;
            case 3: date[i - 5] = QString("WED"); break;
            case 4: date[i - 5] = QString("THU"); break;
            case 5: date[i - 5] = QString("FRI"); break;
            case 6: date[i - 5] = QString("SAT"); break;
            case 7: date[i - 5] = QString("SUN"); break;
        }
    }

    for (int i = 9; i < 13; i++)
        weatherIcon[i - 9] = holdings[i];

    for (int i = 20; i < 24; i++)
    {
        if (convertData == true)
        {
            char tempHold[32];
            double tTemp = holdings[i].toDouble();
            double nTemp = (double)(5.0/9.0)*(tTemp - 32.0);
            sprintf (tempHold, "%d", (int)nTemp);
            holdings[i] = tempHold;
        }
        highTemp[i - 20] = holdings[i];
    }

    for (int i = 40; i < 44; i++)
    {
        if (convertData == true)
        {
            char tempHold[32];
            double tTemp = holdings[i].toDouble();
            double nTemp = (double)(5.0/9.0)*(tTemp - 32.0);
            sprintf (tempHold, "%d", (int)nTemp);
            holdings[i] = tempHold;
        }
        lowTemp[i - 40] = holdings[i];
    }

    for (int i = 15; i < 19; i++)
        weatherType[i - 15] = holdings[i];
}

void WeatherData::weatherTimeout()
{
     stopProcessing = true;
     return;
}
