/* -*- myth -*- */
/**
 * @file mythcontrols.cpp
 * @author Micah F. Galizia <mfgalizi@csd.uwo.ca>
 * @brief Main header for mythcontrols.
 *
 * Note that the keybindings are fetched all at once, and cached for
 * this host.  This avoids pelting the database everytime the user
 * changes their selection.  This makes a HUGE difference in delay.
 *
 * Copyright (C) 2005 Micah Galizia
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA
 */

#ifndef MYTHCONTROLS_CPP
#define MYTHCONTROLS_CPP

/* QT includes */
#include <qnamespace.h>
#include <qstringlist.h>
#include <qapplication.h>
#include <qbuttongroup.h>

/* MythTV includes */
#include <mythtv/mythcontext.h>
#include <mythtv/mythdialogs.h>
#include <mythtv/themedmenu.h>

using namespace std;

#include "mythcontrols.h"
#include "keylistdialog.h"
#include "keygrabber.h"



/* comments in header */
KeyListDialog::KeyListDialog(MythMainWindow *parent, QString context, bool& ui_ok)
    :MythThemedDialog(parent, "keylist", "controls-", "keylist"),
    contextString(context)
{
    cerr << "CONTEXT: " << contextString << endl;

    /* Nullify keybindings so the deconstructor knows not to delete it */
    key_bindings = NULL;

    boundKeys.setAutoDelete(true);

    /* load up the ui components */
    if ((ui_ok = loadUI()))
    {
        /* for starters, load this host */
        loadHost(gContext->GetHostName());

        /* capture the signals we want */
        connect(KeyList, SIGNAL(itemSelected(UIListBtnTypeItem*)),
                this, SLOT(keyListSelected(UIListBtnTypeItem*)));
    }

}



/* comments in header */
KeyListDialog::~KeyListDialog() { delete key_bindings; }



/* comments in header */
bool KeyListDialog::loadUI()
{
    /* the return value of the method */
    bool retval = true;

    /* Get the UI widgets that we need to work with */
    if ((contextText = getUITextType("context")) == NULL)
    {
        VERBOSE(VB_ALL, "KeyListDialog: Unable to load action_context");
        retval = false;
    }

    if ((container = getContainer("controls")) == NULL)
    {
        VERBOSE(VB_ALL, "KeyListDialog:  No controls container in theme");
        retval = false;
    }
    else if ((KeyList = getUIListBtnType("keylist")) == NULL)
    {
        VERBOSE(VB_ALL, "KeyListDialog:  No keylist in theme");
        retval = false;
    }
    else {
        /* focus the context list by default */
        focused = KeyList;
        KeyList->calculateScreenArea();
        KeyList->SetActive(true);
    }

    return retval;
}



void KeyListDialog::keyPressEvent(QKeyEvent *e)
{
    bool handled = false;
    QStringList actions;
    gContext->GetMainWindow()->TranslateKeyPress("Controls", e, actions);

    for (size_t i = 0; i < actions.size() && !handled; i++)
    {
        QString action = actions[i];
        handled = true;

        if (focused == KeyList)
        {
            if (action == "UP")
            {
                KeyList->MoveUp();
            }
            else if (action == "DOWN")
            {
                KeyList->MoveDown();
            }
            else if (action == "PAGEUP")
            {
                KeyList->MoveUp(UIListTreeType::MovePage);
            }
            else if (action == "PAGEDOWN")
            {
                KeyList->MoveDown(UIListTreeType::MovePage);
            }
            else handled = false;
        }
        else handled = false;
    }

    if (!handled)
        MythThemedDialog::keyPressEvent(e);
}

void KeyListDialog::keyListSelected(UIListBtnTypeItem*)
{
    KeyList->blockSignals(true);
    repaintButtonLists();
    KeyList->blockSignals(false);
}

/* method description in header */
void KeyListDialog::repaintButtonLists(void)
{
    /* why am i getting funky colour ? */
    QPixmap pix(container->GetAreaRect().size());
    pix.fill(this, container->GetAreaRect().topLeft());
    QPainter p(&pix);
    if (container)
    {
        container->Draw(&p, 0, 0);
        container->Draw(&p, 1, 0);
        container->Draw(&p, 2, 0);
        container->Draw(&p, 3, 0);
        container->Draw(&p, 4, 0);
        container->Draw(&p, 5, 0);
        container->Draw(&p, 6, 0);
        container->Draw(&p, 7, 0);
        container->Draw(&p, 8, 0);
    }
    p.end();

    bitBlt(this, container->GetAreaRect().left(),
           container->GetAreaRect().top(),
           &pix, 0, 0, -1, -1, Qt::CopyROP);
}




/* comments in header */
QString KeyListDialog::getCurrentContext(void) const
{
    return contextString;
}



/* comments in header */
void KeyListDialog::loadHost(const QString & hostname)
{

    /* create the key bindings and the tree */
    key_bindings = new KeyBindings(hostname);

    KeyList->Reset();

    QStringList *actions = key_bindings->getActions(contextString);
    QStringList *global_actions = key_bindings->getActions(GLOBAL_CONTEXT);

    boundKeys.clear();
    for (size_t i = 0; i < actions->size(); i++)
    {
        QString action = (*actions)[i];
        QStringList keys = key_bindings->getActionKeys(contextString, action);

        for (size_t j = 0; j < keys.size(); j++)
        {
            QString key = keys[j];

            QString *v = boundKeys.find(key);
            if (!v) 
            {
                v = new QString(key + " => " + action);
                boundKeys.insert(key, v);
            }
            else
            {
                *v += ", " + action;
            }
        }
    }

    QDict<QString> globalKeys;
    globalKeys.clear();
    for (size_t i = 0; i < global_actions->size(); i++)
    {
        QString action = (*global_actions)[i];
        QStringList keys = key_bindings->getActionKeys(GLOBAL_CONTEXT, action);

        for (size_t j = 0; j < keys.size(); j++)
        {
            QString key = keys[j];

            QString *v = globalKeys.find(key);
            if (!v)
            {
                v = boundKeys.find(key);
                if (!v)
                {
                    v = new QString(key + " => " + action);
                    globalKeys.insert(key, v);
                }
            }
            else
                *v += ", " + action;
        }
    }

    for (QDictIterator<QString> it(globalKeys); it.current(); ++it)
    {
        QString key = it.currentKey();
        QString *v = it.current();
        boundKeys.insert(key, v);
    }

    QStringList keys;

    for (QDictIterator<QString> it(boundKeys); it.current(); ++it)
    {
        QString key = it.currentKey();
        QString *v = it.current();

        QString keydesc = "3 ";
        if (key.length() == 1)
        {
            switch (key[0].category())
            {
                case QChar::Letter_Uppercase:
                    keydesc = "2 ";
                    break;
                case QChar::Number_DecimalDigit:
                    keydesc = "1 ";
                    break;
            }
        }
        else if (key.find("+", 1) != -1)
            keydesc = "4 ";

        keydesc += *v;

        keys.push_back(keydesc);
    }
    keys.sort();

    UIListBtnTypeItem *item;
    for (size_t i = 0; i < keys.size(); i++) 
    {
        item = new UIListBtnTypeItem(KeyList, keys[i].mid(2));
    }
}


/* comments in header */
void KeyListDialog::deleteKey()
{
}

#endif /* MYTHCONTROLS_CPP */
