// Compile with "gcc -D_GNU_SOURCE -o eas-detect eas-detect.c -lm".
// You need the -D_GNU_SOURCE to keep round from bitching, and the -lm
// to define the math functions.  And w/o the stdlib & math inclusions,
// you get warnings about the types of exit and of sin/cos, respectively.
// I just -love- C's stupid mishmash of obsolete buggy interfaces; don't you?

static char version_msg[] =
"eas-detect v1.0";

static char brief_usage_msg[] =
"\nusage: eas-detect [-a alpha] [-b dB_limit] [-f filter_initial] [-q] [-e] [-s] [-uh]\n";

static char use_u[] =
"Use '-u' or '-h' to see the complete set of options.\n";

static char usage_msg[] =
"Detect either EAS alerts or dead silence in recordings, for use\n"
"by scripts that notify about and/or reschedule damaged recordings.\n"
"\n"
"Expects mono 16-bit 8 kHz PCM with -no- wav header---just the raw samples.\n"
"This expects to be used in a filter chain, so data must be provided via stdin.\n"
"One way to get this from an arbitrary input is (in bash) something like:\n"
"mplayer -really-quiet -noconsolecontrols -nojoystick -nolirc -nomouseinput -vc dummy -vo null -af resample=8000:0:0,channels=1 -ao pcm:nowaveheader:file=/dev/fd/3 INPUTFILE 3>&1 1>&2\n"
"\n"
"Emits output on stdout so it's easy to capture via such scripts.\n"
"May also emit debugging output there for the same reason.\n"
"The mere presence of output does -not- necessarily indicate\n"
"a problem; for that, you should be using the exit value:\n"
"   4  heard an EAS\n"
"   8  heard silence\n"
"  12  heard both\n"
"   0  neither\n"
"  -1  couldn't open stdin\n"
"  -2  error in command arguments\n"
"Note that if you can't get 12 unless both -e and -s are specified,\n"
"since we would otherwise have exited at one or the other event.\n"
"\n"
"Arguments:\n\n"
" Silence detection.  -a, -b, and -f are fully discussed in the comments in the code, q.v.\n"
" Note that absolutely no sanity checking of these inputs is performed.  You have been warned.\n"
"  -a   alpha:  the exponential decay factor\n"
"  -b   dB_limit:  dB below the loudest sound ever in this stream at which we declare silence if it persists\n"
"  -f   filter_initial:  starting value of the exponential filter, which only matters soon after initialization\n\n"
" Whether to quit at the first sign of trouble or not.  Quitting early allows\n"
" early notification; reading the whole stream can be useful for debugging.\n"
"  -e   don't quit at the first EAS alert detected; instead, listen to the whole stream\n"
"  -s   don't quit at the first silence detected; instead, listen to the whole stream\n\n"
" Verboseness.\n"
"  -q   disable debugging output of the structure and characteristics of EAS tonebursts"
"\n";

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <math.h>

int QUIET = 0;

#define EXIT_ALERT   4
#define EXIT_SILENCE 8

// +++ Defining the characteristics of silence.

// We detect silence using an exponentially-smoothed filter, aka an exponential moving average.
// Each block's samples get averaged and used as a single input to the filter.  If the source
// suddenly goes silent, the output of the filter will asymptotically approach zero.  When it
// crosses our silence threshold, we know it's been silent too long.
//
// It's probably best to decide first how quiet is too quiet, and to set your DB_LIMIT there.
// This should be -substantially- above your noise floor!  Since very few programs ever spend
// significant time in very quiet periods, it's safe to be probably 6-12 dB off the floor and
// you -still- won't false-trigger, because after all the source would have to be there quite
// a long time for the filter to get down to there.  How long?  Depends on alpha.  Since this
// is an exponential filter, doubling alpha means we fall off twice as fast, hence halving the
// time it takes to go from an audio->silence transition before we drop below the threshold
// set by DB_LIMIT.  Similarly, adding 3 to DB_LIMIT (which is a doubling of the actual signal
// level) will -also- halve the time it takes to get down to the limit; the issue there is that
// setting it too high means that some sources might legitimately be spending substantial time
// at that level and you'll false-trigger.  [Note that it'd be tricky to establish DB_LIMIT by
// trying to figure out how quiet the source is "typically", because a source that's silent
// from the beginning will have us establish that level as "not silence" and we'll miss the
// very thing we're trying to detect.  Perhaps we could establish this by setting this only
// when surrounded by louder signals (hence -not- setting it from a long silent stretch),
// but we aren't that smart yet.]
//
// For our current values of about 39 blocks/second (e.g., Fs / BLOCKSIZE or 8000 / 205), an
// alpha of 0.0030 means that it takes around 18 seconds from the sudden cessation of audio
// to fall about 24 dB, around 36 seconds to fall 34 dB, and around 63 seconds to fall about
// 44 dB.  Each 10 dB of dropoff is roughly equivalent to losing an entire decimal place in
// the floating-point values coming out of the filter, e.g., if your original source is at
// about -6 dB (hence never going above about 0.25), then the 24 dB point corresponds to
// filter outputs below 0.001 (e.g., 0.000999 or below); 34 dB below 0.0001; and 44 dB
// below 0.00001.  It's probably not advisable to take it much below that, for three reasons:
// (a) You'll run out of precision unless you switch from floats to doubles
// (b) Your tuner or source may have hum or buzz above this level anyway, in which case
//     even the total loss of audio from the headend won't give you anything quieter than
//     this, and 
// (c) Even if it -could-, the fact that we're downsampling the audio from whatever the
//     tuner was providing to mono 16-bit 8 kHz PCM means our signal-to-noise ratio isn't
//     much better than 50 dB -anyway-.  (This is also why we're using floats instead of
//     doubles everywhere else---after all, there are only 2^16 different possible values
//     available to any given sample.)
// Note that, if your input source is very quiet to begin with, setting too high a value
// for DB_LIMIT means we -still- can't get down to there, because there just isn't that
// much headroom available---remember that DB_LIMIT is the difference from the -loudest-
// moment, not the difference from 0 db.  (You get about 50 dB -total-, and that's assuming
// your source's loudest audio is at 0 dB, but running that hot risks clipping.)

int CONTINUE_PAST_FIRST_SILENCE = 0;	// If zero, exit as soon as we think we've found a long-enough section of silence.
float ALPHA = 0.001;			// Exponential decay factor.  See discussion above.
float DB_LIMIT = 24;			// dB below the loudest sound ever in this stream at which we declare silence if it persists.
float FILTER_INITIAL = 0.25;		// Starting value of the filter.  We don't want it to start at 0, because then we'll instantly
					// declare silence.  We -could- just start it at 1.0, but then (given current value of alpha
					// and DB_LIMIT) it'd take at least two minutes to fall if things start out silent and stay
					// that way, and more than four minutes with a limit of, say, 44 dB.  This is currently set
					// to -6 dB, which seems reasonable.  It'll then track the source, so this only matters at
					// initialization.
// ---

// +++ Defining the characteristics of an EAS alert.

// Good references on what an EAS alert looks like:
// http://en.wikipedia.org/wiki/Emergency_Alert_System
// http://en.wikipedia.org/wiki/Specific_Area_Message_Encoding
// http://law.justia.com/us/cfr/title47/47-1.0.1.1.10.2.233.1.html

// A "burst" is a single contiguous pile of tones at the right frequency.
// An "alert" is three long bursts, followed by three short bursts.
// These are critically dependent upon the sample rate and the blocksize,
// since they basically measure the duty cycle of the tonebursts via
// how many contiguous blocks they're detected in.

#define LONG_BURST_LENGTH  30		// (I've typically seen 40..43)  One of the three long bursts at the beginning must be this many blocks long.
#define SHORT_BURST_LENGTH  8		// (I've typically seen 12..13)  One of the three short bursts at the end must be this many blocks long, but not as long as LONG_BURST_LENGTH.
#define INTERBURST_LENGTH_MIN  30	// (I've typically seen 34..35)  The gap between any two bursts must be at least this long.   (These values are for the gaps between any two long or any two short bursts.)
#define INTERBURST_LENGTH_MAX  50	// (I've typically seen 44)      The gap between any two bursts must be no longer than this.  (Note that the gap between the first three and the last three was 44 in my test.)

#define TONE_THRESHOLD 2.0		// Threshold for Goertzel filter's output to determine if the tone is there.  Note that this is -not- on the same scale as the samples!  (Their maximum value is only 1.0)

int CONTINUE_PAST_FIRST_EAS = 0;	// If zero, exit as soon as we think we've found the first alert.  This means that monitoring programs get our results in a timely fashion---within a few seconds of the alert---
					// whether they're using our return value or scanning our output.  Otherwise, we have to make sure that our output isn't buffered by anything (including pipes, etc) or it might
					// not be received until the input stream is closed at the end of the timeslot.  That needlessly decreases the time available to reschedule if the alert was near the beginning.
					// Setting this to zero might be useful in debugging, however, if a recording is thought to have more than one alert, or while testing a state-machine implementation.
#define FINISH_DELAY 500		// If we've seen an alert, how many more blocks to wait after the tone -finishes- before we actually exit, assuming CONTINUE_PAST_FIRST_EAS is false.
					// Mostly for debugging, so we can see -all- the bursts in an alert, instead of exiting at the very first one, in the case where we don't have a true state machine.
					// If there aren't this many left in the stream, we'll just exit at the end of the stream.
// ---

// +++ Defining the characteristics of an EAS toneburst, which makes up part of an alert.

// We detect EAS tonebursts using a Goertzel filter.
// http://www.embedded.com//showArticle.jhtml?articleID=9900722 has a good writeup.

#define PI 3.14159
#define BLOCKSIZE 205   // Block size of the virtual FFT.  Keep this proportional to Fs.
#define Fs 8000.0	// Sample rate, in Hz.  Note that you can't set this arbitrarily low, due to aliasing...

#define NUMTONES 2
float freqs[NUMTONES] = { 1562.5, 2083.3 };	// The actual tones that make up the burst.
// ---

float cw[NUMTONES], sw[NUMTONES], coeff[NUMTONES];	// Precomputed coefficients, set by initGoertzel and never written again.

void initGoertzel(void) {				// Precomputes coefficients.
  int i;
  float k, W;

  for (i = 0; i <  NUMTONES; i++) {
    k = (int) (0.5 + (BLOCKSIZE * freqs[i] / Fs));
    W = (2 * PI / BLOCKSIZE) * k;			// Go on, optimizer, do yo stuff.

    cw[i]    = cos(W);
    sw[i]    = sin(W);
    coeff[i] = 2 * cw[i];
  }
}

void toneBlock(float *buf, float *mags) {		// Checks for presence of each frequency.  Reads buf, writes mags.
  int t, f;
  float q0, q1, q2;
  float real, imag, mag;

  for (f = 0; f < NUMTONES; f++) {
    q1 = 0;
    q2 = 0;
    for (t = 0; t < BLOCKSIZE; t++) {
      q0 = coeff[f] * q1 - q2  + buf[t];
      q2 = q1;
      q1 = q0;
    }
    real = (q1 - q2 * cw[f]);
    imag = q2 * sw[f];
    mag  = real * real + imag * imag;
    mags[f] = mag;
  }
}

float gmin, gmax;				// gmin is really only for debugging, but cheap to compute...
float levelBlock(float *buf) {			// Checks whether this block seems too silent.
  int t;
  float min = 0, max = 0, sum = 0;
  float sample, average;
  for (t = 0; t < BLOCKSIZE; t++) {		// Minima & maxima for this block.  (Minima is really only for debugging.)
    sample = fabsf(buf[t]);
    min = (sample < min) ? sample : min;
    max = (sample > max) ? sample : max;
    sum += sample;
  }
  gmin = (min < gmin) ? min : gmin;		// Global minima & maxima across all blocks.  (Ditto.)
  gmax = (max > gmax) ? max : gmax;
  average = sum / BLOCKSIZE;
  return average;
}

int main(int argc, char *argv[]) {
  FILE *fi;
  int f, t;			// Random iteration variables.
  int c, bad_options = 0;	// For getopt.

  // Parse arguments.
  while ((c = getopt(argc, argv, "a:b:f:qesuh")) != -1)
    switch (c) {
    case 'q': QUIET++;                       break;
    case 'e': CONTINUE_PAST_FIRST_EAS++;     break;
    case 's': CONTINUE_PAST_FIRST_SILENCE++; break;
    case 'a':
      if (sscanf(optarg, "%g", &ALPHA) != 1) bad_options++;
      break;
    case 'b':
      if (sscanf(optarg, "%g", &DB_LIMIT) != 1) bad_options++;
      break;
    case 'f':
      if (sscanf(optarg, "%g", &FILTER_INITIAL) != 1) bad_options++;
      break;
    case 'u': 
    case 'h':
      printf("%s\nCurrent defaults:  -a = %g, -b = %g, -f = %g\n\n%s\n",
	     usage_msg, ALPHA, DB_LIMIT, FILTER_INITIAL, version_msg);
      return 0;
    case '?': 
      bad_options++;
    }
  if (optind != argc) {
    fprintf(stderr, "No arguments are allowed; only switches and their arguments.\n");
    bad_options++;
  }
  if (bad_options) {
    fprintf(stderr, "%s%s\n%s\n", brief_usage_msg, use_u, version_msg);
    return -2;
  }

  // State variables for silence detection.
  int det_silence = 0;		// Have we ever heard silence at all?
  int in_silence  = 0;		// Are we currently in a silent portion?
  float avg;			// Average level of this block.
  float s = FILTER_INITIAL;	// Start out with the filter at some reasonable initial level.
  float limit;			// The actual limit (as a level, not as dB) below which the filter cannot drop before we declare silence.
				// I could save a trivial amount of computation by updating this in levelBlock iff gmax has changed, instead of every time around the main loop, but oh well.

  // State variables for tone detection and alert recognition.
  int det_alert = 0;		// Have we ever seen an alert at all?
  int tonePresent[NUMTONES];	// I suppose this should really be a bitvector; for realistic numbers of tones it'd make the "if tonePresent" stuff later trivial.
  int blocknumber = 0;		// Blocknumber of this sample.  8k mono wav's are about 15K/sec or 56meg/hr which should fit an int fine, and divide by BLOCKSIZE and it fits even better (~141K blocks/hr or ~39 blocks/sec).
  int alert_start = 0;		// If nonzero, this is the blocknumber at which we saw the very start of the alert (e.g,. the blocknumber of the first burst detected).  [If >1 alert (unlikely!) it's the -last- one.]
  int burst_start = 0;		// If nonzero, this is the blocknumber at which the burst currently being processed was first detected.
  int burst_end = 0;		// Used below when deciding whether to exit early after an alert.
  int total_bursts = 0;		// Number of actual bursts we saw.  Should be 6 for a single, well-formed alert.
  int long_bursts_seen = 0;	// How many long bursts have we seen?  [Currently, if we've seen even one, we'll then accept short ones; a real FSM would insist on 3 long bursts, and, only then, 3 short ones.]
  int short_bursts_seen = 0;	// How many short bursts have we seen?
  short sample;			// Endianness and word-size alert!

  // State variables for handling blocks of data.
  float buf[BLOCKSIZE];		// Holds one block of samples.
  float mags[NUMTONES];		// Magnitudes computed from that block.

  // Initialization of I/O and coefficients.
  fi = stdin;
  if (fi == NULL) {
    printf("Can't open stdin.\n");
    exit(-1);
  }

  initGoertzel();		// Precompute coefficients.

  // Note!  gmax MUST NOT start at zero!  Doing so effectively kills the silence detector, since a program that starts out dead silent
  // can never get DB_LIMIT below that value!  On the other hand, starting out at our guess of a peak level (namely FILTER_INITIAL)
  // isn't such a great idea, either, since anything that's relatively quiet throughout (but not a malfunction)---or any source that
  // happens to be quite a bit quieter than whatever we've declared to be our peak in FILTER_INITIAL---will cause us to spuriously
  // declare silences where there aren't any.  But we don't want gmax to start too low---after all, we -must- have at least DB_LIMIT
  // available -below- gmax and -above- the true noise floor of the signal, or even a truly silent recording won't be detected as such,
  // because the filter can't fall low enough no matter how long we wait.  As a compromise, therefore, we set gmax to 1/8 of FILTER_INITIAL,
  // which is equivalent to setting it 2^3 or 9 dB below FILTER_INITIAL, which itself defaults to 1/4 (6 dB) of the absolute loudest any
  // PCM file might be.  This means that no recording will ever have a claimed gmax that is less than 6 + 9 = 15 dB from the very top
  // unless we're called with some screwy value of -f (and even -f 1.0 means gmax is still 9 dB down).  This gives us, in the default
  // case, about 50 - 15 = 35 dB to play with in setting DB_LIMIT, which should be adequate.  (Note:  to simulate an all-silence failure,
  // just pipe /dev/zero into our stdin.)
  gmax = FILTER_INITIAL / 8;

  // Main loop.
  while (! feof(fi)) {

    // Read in a block of samples to process.
    for (t = 0; t < BLOCKSIZE; t++)		// Note endianness/size issues (in the line below) with just casting the result of fread into &sample.
      buf[t] = fread((char*) &sample, 2, 1, fi) ? ((float) sample) / 65536.0 : 0.0;	// Zero-pad if we run off the end of the block.

    // See if the block is silent.
    avg = levelBlock(buf);				// printf("%f%s",   avg, ((blocknumber + 1) % 25 == 0) ? "\n" : " ");
    s += ALPHA * (avg - s);				// printf("%f%s",     s, ((blocknumber + 1) % 25 == 0) ? "\n" : " ");
    limit = gmax / (1 << (int) (DB_LIMIT / 3));		// printf("%f%s", limit, ((blocknumber + 1) % 25 == 0) ? "\n" : " ");
    if (s < limit) {
      if (! in_silence) {				// Have we noticed this silence before?
	det_silence = in_silence = EXIT_SILENCE;	// No:  notify about it.
	float timecode = (float) blocknumber / ((float) Fs / (float) BLOCKSIZE);	// Making this a macro, or some function using asprintf, isn't worth it when it's done only twice and called once during execution.
	int min = timecode / 60;
	int sec = round(timecode - (min * 60));
	printf("Silence detected at %i:%02i\n", min, sec);
	fflush(NULL);
	if (! CONTINUE_PAST_FIRST_SILENCE)
	  break;
      }
    } else {
      in_silence = 0;					// Whether or not we were ever in a silent period, we're out of it now, so re-arm the alerting mechanism.
    }

    // Find tones in the block of samples.
    toneBlock(buf, mags);	// buf is only read; mags is updated with the various magnitudes.

    // Threshold to detect if each tone is present or not.
    for (f = 0; f < NUMTONES; f++)
      tonePresent[f] = (mags[f] > TONE_THRESHOLD) ? 1 : 0;

    // Decide whether or not we're currently in a burst, so we can later decide if we're in a full-blown alert.
    //
    // Note that I -could- whip up a whole little state machine to make sure we have three long and three short bursts
    // with the correct timing relationships, but that seems overkill until I find a false positive in my corpus.
    // Until then, just look for a single burst that is at least as long as the minimum duration and, if we find
    // one, assume that means we've got an alert.  This is also safer (at least until I see a false positive),
    // because it handles alerts where one of the bursts got truncated or mangled or whatever, and it's safer
    // to err on the side of requesting a rerecording than to miss one that got smashed.  --- Okay, that's still
    // -slightly- too forgiving:  there are plenty of occurrences in my corpus that have at least one burst that's
    // longer than the minimum length of a -short- burst, yet have no long bursts and thus couldn't be an EAS.
    // So now what we do is to ensure that we don't decide this is -really- an EAS until we've seen at least one
    // long burst (where our "minimum" for that is still substantially shorter than any real long burst we've seen,
    // but longer than the longest spurious one we've seen).  This still isn't as strict as we could be (e.g.,
    // insist on all three long bursts, followed by all three short bursts, and all of them within a certain total
    // amount of time), but it's probably good enough to avoid falsing, while decreasing our risk of missing a real
    // EAS due to being too strict.

    // A burst starts when BOTH of the tones are present simultaneously.  (Lots of theme music tends to have only
    // one or the other.)  Note that many true bursts -do- seem start with 12 blocks or so (using our definition
    // of BLOCKSIZE above) of the 1562.5 Hz tones only, or even up to 20 blocks of either 1562.5 xor 2083.3, but
    // then they actually -do- have both tones for at least 40 blocks (modulo the occasional one-block dropout
    // back to a single tone).
    //
    // So the current detection strategy is that a burst starts only when we see both tones.  Once we've declared
    // that it's started, we won't drop out even if one of the tones goes away, until they both do.  If this turns
    // out in practice to be too forgiving, we'll insist that the dropouts be of limited duration (e.g., only one
    // or two blocks).  In practice, the bursts I've seen tend to have only one-block dropouts, but even bothering
    // to check is too much until I get falsing---I'd rather notice a false positive, and only -then- back off, than
    // get a false negative (which I'll -never- notice unless I'm extremely lucky) and not know we're being too strict.
    //
    // Yeah, yeah, should loop over NUMTONES and not assume exactly 2 in both of the tests below.  So sue me.
    if (tonePresent[0] && tonePresent[1]) {					// If we see BOTH tones simultaneously, this is the start of a burst.
      if (burst_start == 0)
	burst_start = blocknumber;
    } else if (burst_start != 0 && (! (tonePresent[0] || tonePresent[1]))) {	// If we're already in a burst and we see either tone, we're still in the burst.
      // Okay, we fell out of the burst.  See if it's good or a runt.
      int burst_length = blocknumber - burst_start;
      int good_burst = 0;

      if (burst_length >= LONG_BURST_LENGTH) {
	long_bursts_seen++;
	good_burst = 1;
      }
      else if (burst_length >= SHORT_BURST_LENGTH && (long_bursts_seen > 0)) {	// We can't see a short burst until we've seen at least one long burst.  (An FSM would say, "until we've seen -all three- long bursts.")
	short_bursts_seen++;
	good_burst = 1;
      }
      else if (! QUIET) {							// If we're not quiet, we'll have been printing "11 01 10" etc, so visually separate this from the next possible runt or good burst.
	printf(" runt  [%i]\n", burst_start);
      }

      if (good_burst) {								// This burst was long enough, e.g., not a runt.
	burst_end = blocknumber;
	total_bursts++;

	// We're declaring that even a single long-enough burst starts an alert.  A real FSM would insist on 3 long, followed by 3 short, with the right gaps, and no more than the right total length.  Too strict.
	if (long_bursts_seen == 1) alert_start = burst_start;			// The alert starts at the start of the first long burst.
	det_alert = EXIT_ALERT;

	// Get the news out about an alert ASAP, since we'll want to reschedule immediately (probably while we're still
	// near the beginning of the timeslot) rather than only finding out a few minutes after the end of the timeslot.
	// See comments above at CONTINUE_PAST_FIRST_EAS for why this would particularly matter if that was false.
	float timecode = (float) burst_start / ((float) Fs / (float) BLOCKSIZE);	// See comment in silence-detector above.
	int min = timecode / 60;
	int sec = round(timecode - (min * 60));
	printf(" [%i]  %i:%02i\n", burst_length, min, sec);
	fflush(NULL);
      }
      burst_start = 0;	// Reset to detect the next burst.
    }
    // This stuff runs every cycle, whether or not we're currently currently in a burst.
    // (We can tell if we are by whether burst_start is nonzero, but we won't know here if it's a runt, a short burst, or a long burst.)
    if ((! QUIET) && burst_start != 0)
      printf("%d%d ", tonePresent[0], tonePresent[1]);

    // If we're supposed to exit almost immediately after seeing an alert, do so.
    // Note that we don't start counting until the end of a burst, so a recording
    // that's one continuous burst won't have us exit until the very end---but that's
    // not really an EAS alert, now, is it?
    if ((! CONTINUE_PAST_FIRST_EAS) && det_alert && (blocknumber - burst_end) > FINISH_DELAY)
      break;

    // Next block!
    blocknumber++;
  }

//printf("\n[%f %f]\n\n", gmin, gmax);

  if (total_bursts) printf("%i total bursts (%i long, %i short), starting at block %i\n", total_bursts, long_bursts_seen, short_bursts_seen, alert_start);
  return det_alert | det_silence;
}
