/*
 * This file is part of Flowplayer, http://flowplayer.org
 *
 * By: Anssi Piirainen, <support@flowplayer.org>
 * Copyright (c) 2008, 2009 Flowplayer Oy

 * H.264 support by: Arjen Wagenaar, <h264@code-shop.com>
 * Copyright (c) 2009 CodeShop B.V.
 *
 * Released under the MIT License:
 * http://www.opensource.org/licenses/mit-license.php
 */

// Todo: seek twice ???

package org.flowplayer.pseudostreaming {
import org.flowplayer.model.ClipEvent;
import org.flowplayer.controller.NetStreamControllingStreamProvider;
import org.flowplayer.model.ClipEventType;
import org.flowplayer.model.Plugin;
import org.flowplayer.model.PluginModel;
import org.flowplayer.util.PropertyBinder;
import org.flowplayer.view.Flowplayer;

import flash.events.NetStatusEvent;
import org.flowplayer.model.Clip;
import flash.net.NetStream;

    /**
     * @author api
     */
public class PseudoStreamProvider extends NetStreamControllingStreamProvider implements Plugin {
    private var _bufferStart:Number;
    private var _config:Config;
    private var _fileWithKeyframeInfo:String;
    private var _serverSeekInProgress:Boolean;
    private var _startSeekDone:Boolean;
    private var _model:PluginModel;

    /**
     * Called by the player to set my config.
     */
    override public function onConfig(model:PluginModel):void {
        _model = model;
        _config = new PropertyBinder(new Config(), null).copyProperties(model.config) as Config;
    }

    /**
     * Called by the player to set the Flowplayer API.
     */
    override public function onLoad(player:Flowplayer):void {
        log.info("onLoad, registering metadata listener");
        _model.dispatchOnLoad();
    }

    override protected function getClipUrl(clip:Clip):String {
        log.info("getClipUrl, " + clip.completeUrl);
        return appendQueryString(clip.completeUrl, 0);
    }

    override protected function doLoad(event:ClipEvent, netStream:NetStream, clip:Clip):void {
        log.info("doLoad");
        _bufferStart = clip.start;
        _startSeekDone = false;
        super.doLoad(event, netStream, clip);
    }

    private function isNewFile(clip:Clip):Boolean {
        log.info("isNewFile");
        return clip.url != _fileWithKeyframeInfo;
    }

    override protected function doSeek(event:ClipEvent, netStream:NetStream, seconds:Number):void {
        log.info("doSeek");
        var target:Number = clip.start + seconds;
        if (isInBuffer(target)) {
            log.debug("seeking inside buffer, target " + target + " seconds");
            netStream.seek(target - _bufferStart);
        } else {

           log.debug("doSeek: seeking, target " + target + " seconds");
           serverSeek(netStream, target);
        }
    }

    override public function get bufferStart():Number {
        if (! clip) return 0;
        return _bufferStart - clip.start;
    }

    override public function get bufferEnd():Number {
        if (! netStream) return 0;
        if (! clip) return 0;
        return bufferStart + netStream.bytesLoaded/netStream.bytesTotal * (clip.duration - bufferStart);
    }

    override protected function getCurrentPlayheadTime(netStream:NetStream):Number {
        if (! clip) return 0;
        var value:Number = _bufferStart+netStream.time;

        return value < 0 ? 0 : value;
    }

    override public function get allowRandomSeek():Boolean {
          return true;
    }

    private function isInBuffer(seconds:Number):Boolean {
        log.info("isInBuffer");
        return bufferStart <= seconds - clip.start && seconds - clip.start <= bufferEnd;
    }

    private function serverSeek(netStream:NetStream, seconds:Number, setBufferStart:Boolean = true, silent:Boolean = false):void {
        if (setBufferStart) {
            _bufferStart = Math.round(seconds);
        }
        var requestUrl:String = appendQueryString(clip.completeUrl , seconds);
        log.debug("doing server seek, url " + requestUrl);
        if (! silent) {
            _serverSeekInProgress = true;
        }
        netStream.play(requestUrl);
    }

    private function appendQueryString(url:String, start:Number):String {
        log.info("appendQueryString, " + url + " " + start);
        return url + _config.queryString.replace("${start}", start == 0 ? 0 : Math.round(start));
    }

    override protected function onMetaData(event:ClipEvent):void {
        if (_startSeekDone) {
            return;
        }

        log.info("received metaData for clip" + Clip(event.target));
        log.debug("clip file is " + clip.url);
        if (isNewFile(event.target as Clip)) {
            log.info("new file, creating new keyframe store");
        }
        _fileWithKeyframeInfo = Clip(event.target).url;

        clip.dispatch(ClipEventType.START, pauseAfterStart);
        if (pauseAfterStart) {
            clip.dispatch(ClipEventType.PAUSE);
        }

        // at this point we seek to the start position if it's greater than zero
        log.debug("onMetaData: seeking to start, pausing after start: " + pauseAfterStart);
        if (clip.start > 0) {
            serverSeek(netStream, clip.start, true, true);
            _startSeekDone = true;

        } else if (pauseAfterStart) {
            netStream.seek(0);
            pauseAfterStart = false;
        }
    }

    override protected function canDispatchBegin():Boolean {
        log.info("canDispatchBegin");
        if (_serverSeekInProgress) return false;
        if (clip.start > 0 && ! _startSeekDone) return false;
        return true;
    }

    override protected function onNetStatus(event:NetStatusEvent):void {
        log.info("onNetStatus: " + event.info.code);
        if (event.info.code == "NetStream.Play.Start") {



            log.debug("started, will pause after start: " + pauseAfterStart);
            // we need to pause here because the stream was started when server-seeking to start pos
            if (paused || pauseAfterStart) {
                log.info("started: pausing to pos 0 in netStream");
                netStream.seek(0);
                pause(null);
                if (_startSeekDone) {
                    pauseAfterStart = false;
                }
            }
            // at this stage the server seek is in target, and we can dispatch the seek event
            if (_serverSeekInProgress) {
                _serverSeekInProgress = false;
                clip.dispatch(ClipEventType.SEEK, seekTarget);
            }
        }
    }

    public function getDefaultConfig():Object {
        log.info("getDefaultConfig");
        return null;
    }
}
}
