#include <stdio.h>
#include <string.h>
#include <stdint.h>

#ifndef ARCH_X86_64
#define ARCH_X86_32
#endif

#ifdef ARCH_X86_64
#  define REG_b "rbx"
#  define REG_S "rsi"
typedef int64_t x86_reg;
#else
#  define REG_b "ebx"
#  define REG_S "esi"
typedef int32_t x86_reg;
#endif

#define MM_MMX    0x0001 /* standard MMX */
#define MM_MMX2     0x0002 ///< SSE integer functions or AMD MMX ext
#define MM_3DNOW  0x0004 /* AMD 3DNOW */
#define MM_MMXEXT 0x0002 /* SSE integer functions or AMD MMX ext */
#define MM_SSE    0x0008 /* SSE functions */
#define MM_SSE2   0x0010 /* PIV SSE2 functions */
#define MM_SSE2SLOW 0x40000000 ///< SSE2 supported, but usually not faster
                                  ///< than regular MMX/SSE (e.g. Core1)
#define MM_SSE3SLOW 0x20000000 ///< SSE3 supported, but usually not faster
                                  ///< than regular MMX/SSE (e.g. Core1)
#define MM_3DNOWEXT  0x0020 /* AMD 3DNowExt */
#define MM_SSE3   0x0010 /* SSE3 functions */
#define MM_SSSE3  0x0010 /* SSSE3 functions */
#define MM_SSE4   0x0010 /* SSE4.1 functions */
#define MM_SSE42  0x0010 /* SSE4.2 functions */

/* ebx saving is necessary for PIC. gcc seems unable to see it alone */
#define cpuid(index,eax,ebx,ecx,edx)\
    __asm __volatile\
        ("mov %%"REG_b", %%"REG_S"\n\t"\
         "cpuid\n\t"\
         "xchg %%"REG_b", %%"REG_S\
         : "=a" (eax), "=S" (ebx),\
           "=c" (ecx), "=d" (edx)\
         : "0" (index));

/* Function to test if multimedia instructions are supported... */
int mm_support1(void)
{
    int rval = 0;
    int eax, ebx, ecx, edx;
    int max_std_level, max_ext_level, std_caps=0, ext_caps=0;
    int family=0, model=0;
    union { int i[3]; char c[12]; } vendor;

#ifdef ARCH_X86_32
    x86_reg a, c;
    __asm__ volatile (
        /* See if CPUID instruction is supported ... */
        /* ... Get copies of EFLAGS into eax and ecx */
        "pushfl\n\t"
        "pop %0\n\t"
        "mov %0, %1\n\t"

        /* ... Toggle the ID bit in one copy and store */
        /* to the EFLAGS reg */
        "xor $0x200000, %0\n\t"
        "push %0\n\t"
        "popfl\n\t"

        /* ... Get the (hopefully modified) EFLAGS */
        "pushfl\n\t"
        "pop %0\n\t"
        : "=a" (a), "=c" (c)
        :
        : "cc"
        );

    if (a == c)
        return 0; /* CPUID not supported */
#endif

    cpuid(0, max_std_level, vendor.i[0], vendor.i[2], vendor.i[1]);

    if(max_std_level >= 1){
        cpuid(1, eax, ebx, ecx, std_caps);

        family = ((eax>>8)&0xf) + ((eax>>20)&0xff);
        model = ((eax>>4)&0xf) + ((eax>>12)&0xf0);
        if (std_caps & (1<<23))
            rval |= MM_MMX;
        if (std_caps & (1<<25))
            rval |= MM_MMX2
                  | MM_SSE;
        if (std_caps & (1<<26))
            rval |= MM_SSE2;
        if (ecx & 1)
            rval |= MM_SSE3;
        if (ecx & 0x00000200 )
            rval |= MM_SSSE3;
        if (ecx & 0x00080000 )
            rval |= MM_SSE4;
        if (ecx & 0x00100000 )
            rval |= MM_SSE42;
    }

    cpuid(0x80000000, max_ext_level, ebx, ecx, edx);

    if(max_ext_level >= 0x80000001){
        cpuid(0x80000001, eax, ebx, ecx, ext_caps);
        if (ext_caps & (1<<31))
            rval |= MM_3DNOW;
        if (ext_caps & (1<<30))
            rval |= MM_3DNOWEXT;
        if (ext_caps & (1<<23))
            rval |= MM_MMX;
        if (ext_caps & (1<<22))
            rval |= MM_MMX2;
    }

    if (!strncmp(vendor.c, "GenuineIntel", 12) &&
        family == 6 && (model == 9 || model == 13 || model == 14)) {
        /* 6/9 (pentium-m "banias"), 6/13 (pentium-m "dothan"), and 6/14 (core1 "yonah")
* theoretically support sse2, but it's usually slower than mmx,
* so let's just pretend they don't. */
        if (rval & MM_SSE2) rval ^= MM_SSE2SLOW|MM_SSE2;
        if (rval & MM_SSE3) rval ^= MM_SSE3SLOW|MM_SSE3;
    }

    return rval;
}


/* Function to test if multimedia instructions are supported...  */
static int mm_support2(void)
{
    int rval = 0;
    int eax, ebx, ecx, edx;
    int max_std_level, max_ext_level, std_caps=0, ext_caps=0;
    long a, c;

    __asm__ __volatile__ (
                          /* See if CPUID instruction is supported ... */
                          /* ... Get copies of EFLAGS into eax and ecx */
                          "pushf\n\t"
                          "pop %0\n\t"
                          "mov %0, %1\n\t"

                          /* ... Toggle the ID bit in one copy and store */
                          /*     to the EFLAGS reg */
                          "xor $0x200000, %0\n\t"
                          "push %0\n\t"
                          "popf\n\t"

                          /* ... Get the (hopefully modified) EFLAGS */
                          "pushf\n\t"
                          "pop %0\n\t"
                          : "=a" (a), "=c" (c)
                          :
                          : "cc"
                          );

    if (a == c)
        return 0; /* CPUID not supported */

    cpuid(0, max_std_level, ebx, ecx, edx);

    if(max_std_level >= 1){
        cpuid(1, eax, ebx, ecx, std_caps);
        if (std_caps & (1<<23))
            rval |= MM_MMX;
        if (std_caps & (1<<25))
            rval |= MM_MMXEXT | MM_SSE;
        if (std_caps & (1<<26))
            rval |= MM_SSE2;
        if (ecx & 1)
            rval |= MM_SSE3;
        if (ecx & 0x00000200 )
            rval |= MM_SSSE3;
        if (ecx & 0x00080000 )
            rval |= MM_SSE4;
        if (ecx & 0x00100000 )
            rval |= MM_SSE42;
    }

    cpuid(0x80000000, max_ext_level, ebx, ecx, edx);

    if(max_ext_level >= 0x80000001){
        cpuid(0x80000001, eax, ebx, ecx, ext_caps);
        if (ext_caps & (1<<31))
            rval |= MM_3DNOW;
        if (ext_caps & (1<<30))
            rval |= MM_3DNOWEXT;
        if (ext_caps & (1<<23))
            rval |= MM_MMX;
    }

    cpuid(0, eax, ebx, ecx, edx);
    if (       ebx == 0x68747541 &&
               edx == 0x69746e65 &&
               ecx == 0x444d4163) {
        /* AMD */
        if(ext_caps & (1<<22))
            rval |= MM_MMXEXT;
    } else if (ebx == 0x746e6543 &&
               edx == 0x48727561 &&
               ecx == 0x736c7561) {  /*  "CentaurHauls" */
        /* VIA C3 */
        if(ext_caps & (1<<24))
          rval |= MM_MMXEXT;
    } else if (ebx == 0x69727943 &&
               edx == 0x736e4978 &&
               ecx == 0x64616574) {
        /* Cyrix Section */
        /* See if extended CPUID level 80000001 is supported */
        /* The value of CPUID/80000001 for the 6x86MX is undefined
           according to the Cyrix CPU Detection Guide (Preliminary
           Rev. 1.01 table 1), so we'll check the value of eax for
           CPUID/0 to see if standard CPUID level 2 is supported.
           According to the table, the only CPU which supports level
           2 is also the only one which supports extended CPUID levels.
        */
        if (eax < 2)
            return rval;
        if (ext_caps & (1<<24))
            rval |= MM_MMXEXT;
    }
    return rval;
}

int main ( void )
{
    int mm_flags1, mm_flags2;
    mm_flags1 = mm_support1();
    mm_flags2 = mm_support2();
    printf("mm_support1 = 0x%08X\n",mm_flags1);
    printf("mm_support2 = 0x%08X\n",mm_flags2);
    return 0;
}
